package cn.com.duiba.geo.local.common.tire;

import com.google.common.collect.Lists;
import org.apache.commons.lang3.StringUtils;

import java.util.Collections;
import java.util.LinkedList;
import java.util.List;
import java.util.Objects;

/**
 * 字典树
 * 刘瑶
 */
public class TireTree<T extends TireTreeNodeShim> {

    private final TireTreeNode root = new TireTreeNode();

    private int maxLevel = 0;

    private final TireTreeNodePolicy policy;

    public TireTree() {
        this.policy = new DefaultTireTreeNodePolicy();
    }

    public TireTree(TireTreeNodePolicy policy) {
        this.policy = policy;
    }

    public void insert(T node){
        String code = node.getTireTreeKey();
        if(StringUtils.isBlank(code)){
            throw new IllegalArgumentException("字典树策略的Code不能为空");
        }
        int size = policy.charSize();
        char[] chars = code.toCharArray();
        for(char item:chars){
            int index = policy.charIndex(item);
            if(index < 0 || index >= size){
                throw new IllegalArgumentException("字典树策略的存在值域溢出:" + code);
            }
        }
        root.insert(chars,0,policy,node);
        maxLevel = Math.max(maxLevel,chars.length);
    }

    public T find(String code){
        char[] chars = code.toCharArray();
        TireTreeNode node = root.find(chars,0,policy);
        if(Objects.isNull(node)){
            return null;
        }
        return (T)node.getPayload();
    }

    public List<T> findNodeLink(String code){
        char[] chars = code.toCharArray();
        LinkedList<TireTreeNode> nodes = Lists.newLinkedList();
        root.findToList(chars,0,nodes,policy);
        return transform(nodes);
    }

    /**
     * 查询直接的孩子节点
     * @param parentCode 父级节点
     * @param acceptParentNotExist 当前节点不存在，但是子节点存在元素，是否允许继续查询
     * @return 孩子节点集合
     */
    public List<T> findChildNodes(String parentCode,Boolean acceptParentNotExist){
        char[] chars = new char[0];
        if(StringUtils.isNotBlank(parentCode)){
            chars = parentCode.toCharArray();
        }
        TireTreeNode node = root.find(chars,0,policy);
        if(Objects.isNull(node)){
            return Collections.emptyList();
        }
        if(Objects.isNull(node.getPayload()) && !acceptParentNotExist){
            return Collections.emptyList();
        }
        List<TireTreeNode> keys = Lists.newLinkedList();
        node.findInDeep(keys);
        return transform(keys);
    }

    private List<T> transform(List<TireTreeNode> keys){
        List<T> returnList = Lists.newLinkedList();
        for(TireTreeNode node:keys){
            TireTreeNodeShim shim = node.getPayload();
            if(Objects.nonNull(shim)){
                returnList.add((T)shim);
            }
        }
        return returnList;
    }

    public boolean isEmpty(){
        return maxLevel==0;
    }

}
