package cn.com.duiba.geo.local.service.ad;

import cn.com.duiba.geo.local.common.tire.TireTree;
import cn.com.duiba.geo.local.dao.AdministrativeDivisionDao;
import cn.com.duiba.geo.local.domain.entity.AdministrativeDivisionDO;
import cn.com.duiba.geo.local.dto.AdministrativeDivisionDto;
import com.google.common.collect.HashMultimap;
import com.google.common.collect.Lists;
import com.google.common.collect.Multimap;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.core.Ordered;
import org.springframework.core.annotation.Order;
import org.springframework.stereotype.Service;

import javax.annotation.PostConstruct;
import javax.annotation.Resource;
import java.util.Collections;
import java.util.List;

/**
 * @author liuyao
 */
@Slf4j
@Service
@Order(Ordered.LOWEST_PRECEDENCE - 1)
public class AdministrativeDivisionService {

    @Resource
    private AdministrativeDivisionDao administrativeDivisionDao;

    private volatile TireTree<AdministrativeDivisionTreeNode> tireTree = new TireTree<>();

    private volatile Multimap<String,AdministrativeDivisionTreeNode> areaNameMap = HashMultimap.create();

    @PostConstruct
    public void initData(){
        TireTree<AdministrativeDivisionTreeNode> newTireTree = new TireTree<>();
        Multimap<String,AdministrativeDivisionTreeNode> newAreaNameMap = HashMultimap.create();

        List<AdministrativeDivisionDO> list = administrativeDivisionDao.findSimpleAll();
        for(AdministrativeDivisionDO item:list){
            AdministrativeDivisionTreeNode node = new AdministrativeDivisionTreeNode();
            node.setCode(item.getCode());
            node.setName(item.getName());
            node.setLevel(item.getLevel());
            newAreaNameMap.put(item.getName(),node);
            newAreaNameMap.put(simpleName(item.getName()),node);
            newTireTree.insert(node);
        }
        tireTree = newTireTree;
        areaNameMap = newAreaNameMap;
        log.info("行政区划字典树构建完成,总数:"+list.size());
    }

    /**
     * 查询具体地址链路
     */
    public List<AdministrativeDivisionDto> findCodeLink(String code){
        if(!isLegal(code)){
            return Collections.emptyList();
        }
        String key = AdministrativeDivisionTreeNode.analysisCode(code);
        List<AdministrativeDivisionTreeNode> list = tireTree.findNodeLink(key);
        return trasform(list);
    }

    private List<AdministrativeDivisionDto> trasform(List<AdministrativeDivisionTreeNode> nodes){
        List<AdministrativeDivisionDto> returnList = Lists.newArrayList();
        for(AdministrativeDivisionTreeNode node:nodes) {
            AdministrativeDivisionDto dto = new AdministrativeDivisionDto();
            dto.setAdCode(node.getCode());
            dto.setName(node.getName());
            dto.setLevel(node.getLevel());
            returnList.add(dto);
        }
        return returnList;
    }

    /**
     * 简化 省 市 县 名称
     */
    public String simpleName(String name){
        if(StringUtils.isBlank(name)){
            return name;
        }
        if(name.length()<=2){
            return name;
        }
        if(name.contains("自治")){
            return name;
        }
        if(StringUtils.endsWithAny(name,"省","州","市","县","镇")){
            return name.substring(0,name.length()-1);
        }else if(StringUtils.endsWith(name,"街道")){
            return name.substring(0,name.length()-2);
        }else{
            return name;
        }
    }

    public Boolean isLegal(String adcode){
        if(StringUtils.equals("",adcode)){
            return true;
        }
        int length = adcode.length();
        if(length!=6 && length!=9 && length!=12){
            return false;
        }
        try{
            Integer.parseInt(adcode);
        }catch (NumberFormatException e){
            log.debug(adcode+"参数不合法");
            return false;
        }
        return true;
    }

}
