package cn.com.duiba.geo.local.service.point;

import ch.hsr.geohash.GeoHash;
import ch.hsr.geohash.WGS84Point;
import cn.com.duiba.geo.local.GeoBizProperties;
import cn.com.duiba.geo.local.common.tire.TireTree;
import cn.com.duiba.geo.local.dao.GeoPointDao;
import cn.com.duiba.geo.local.domain.entity.GeoPointDO;
import cn.com.duiba.geo.local.dto.AdministrativeDivisionDto;
import cn.com.duiba.geo.local.dto.CoordinateSystem;
import cn.com.duiba.geo.local.dto.GeoInfoDto;
import cn.com.duiba.geo.local.dto.Point;
import cn.com.duiba.geo.local.service.ad.AdministrativeDivisionService;
import cn.com.duiba.geo.local.tool.GpsCoordinateUtils;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.springframework.stereotype.Service;

import javax.annotation.PostConstruct;
import javax.annotation.Resource;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * @author liuyao
 */
@Slf4j
@Service
public class GeoPointService {

    public static final int GEO_HASH_LENGTH = 8;
    private volatile TireTree<GeoPointTireTreeNode> tireTree = new TireTree<>(new GeoPointTireTreePolicy());

    @Resource
    private AdministrativeDivisionService administrativeDivisionService;
    @Resource
    private GeoPointDao geoPointDao;
    @Resource
    private GeoBizProperties geoBizProperties;

    public GeoInfoDto findGeoInfoDto(GeoHash geoHash){

        GeoInfoDto geoInfoDto = new GeoInfoDto();
        String geoHashString = geoHash.toBase32();
        geoInfoDto.setGeoHash(geoHashString);

        WGS84Point point = geoHash.getOriginatingPoint();
        if(GpsCoordinateUtils.isOutOfChina(point.getLatitude(),point.getLongitude(),false)){
            //在国外不返回
            return geoInfoDto;
        }

        GeoPointTireTreeNode node = tireTree.find(geoHashString);
        if(Objects.isNull(node)){
            //降级查询，最近的点
            node = findNearGeoInfoDto(geoHash);
        }
        if(Objects.isNull(node)){
            return geoInfoDto;
        }
        List<AdministrativeDivisionDto> adList = administrativeDivisionService.findCodeLink(node.getAdCode());
        if(adList.isEmpty()){
            return geoInfoDto;
        }
        AdministrativeDivisionDto administrativeDivision = adList.get(adList.size()-1);
        geoInfoDto.setAdLevel(administrativeDivision.getLevel());
        geoInfoDto.setAdCode(administrativeDivision.getAdCode());
        geoInfoDto.setAdNames(adList.stream().map(AdministrativeDivisionDto::getName).collect(Collectors.toList()));

        return geoInfoDto;
    }

    private GeoPointTireTreeNode findNearGeoInfoDto(GeoHash geoHash){

        List<GeoPointTireTreeNode> nodes;
        String parentGeoHash = geoHash.toBase32();
        do{
            parentGeoHash = getParentGeoHash(parentGeoHash);
            nodes = tireTree.findChildNodes(parentGeoHash,true);
            if(parentGeoHash.length()<=1 && nodes.isEmpty()){
                return null;
            }
        }while (nodes.isEmpty() && StringUtils.isNotBlank(parentGeoHash));

        if(nodes.isEmpty()){
            return null;
        }
        GeoHash temp = GeoHash.fromGeohashString(parentGeoHash);

        //获取8个方向的点
        for(GeoHash item:temp.getAdjacent()){
            nodes.addAll(tireTree.findChildNodes(item.toBase32(),true));
        }
        Map<Double,GeoPointTireTreeNode> distanceMap = Maps.newHashMap();

        WGS84Point foo = geoHash.getOriginatingPoint();

        for(GeoPointTireTreeNode node:nodes){
            WGS84Point bar = new WGS84Point(node.getLatitude(),node.getLongitude());
            //正相关距离，不是实际距离
            double distance = distanceInMeters(foo,bar);
            distanceMap.put(distance,node);
        }
        double minDistance = Collections.min(distanceMap.keySet());
        return distanceMap.get(minDistance);
    }

    /**
     * 节约性能，这里只返回一个和两点距离成正相关的量
     */
    private double distanceInMeters(WGS84Point foo,WGS84Point bar){
        return Math.pow((foo.getLatitude() - bar.getLatitude()),2)+Math.pow((foo.getLongitude() - bar.getLongitude()),2);
    }


    private String getParentGeoHash(String geoHash){
        if(StringUtils.isBlank(geoHash)){
            return "";
        }
        return geoHash.substring(0,geoHash.length()-1);
    }

    @PostConstruct
    public synchronized void initData(){
        if(!geoBizProperties.getLoadGeoPoint()){
            return;
        }
        TireTree<GeoPointTireTreeNode> newTireTree = new TireTree<>(new GeoPointTireTreePolicy());
        try{
            List<GeoPointDO> list = geoPointDao.findAll();
            for(GeoPointDO item:list){
                GeoPointTireTreeNode node = new GeoPointTireTreeNode();
                node.setAdCode(item.getAdCode());
                node.setGeoHash(item.getGeoHash());
                node.setLatitude(item.getLatitude());
                node.setLongitude(item.getLongitude());
                newTireTree.insert(node);
            }
            this.tireTree = newTireTree;
            log.info("Geo定位数据加载完成,SIZE:{}",list.size());
        }catch (Exception e){
            log.error("Geo定位点加载异常",e);
        }
    }

    /**
     * 将其他坐标系转成地球坐标系
     * @param coordinateSystem 输入的坐标系
     * @return WGS84Point
     */
    public GeoHash transformGeoHash(WGS84Point point, CoordinateSystem coordinateSystem){

        double latitude = point.getLatitude();
        double longitude = point.getLongitude();

        Point gcj02 = null;
        if(coordinateSystem == CoordinateSystem.WGS84){
            gcj02 = GpsCoordinateUtils.calWGS84ToGCJ02(latitude,longitude);
        }
        if(coordinateSystem == CoordinateSystem.BD09){
            gcj02 = GpsCoordinateUtils.calBD09toGCJ02(latitude,longitude);
        }
        if(Objects.nonNull(gcj02)){
            latitude = gcj02.getLatitude();
            longitude = gcj02.getLongitude();
        }
        return GeoHash.withCharacterPrecision(latitude,longitude,GeoPointService.GEO_HASH_LENGTH);
    }

}
