package cn.com.duiba.goods.center.api.remoteservice.dto.item;

import cn.com.duiba.goods.center.api.remoteservice.util.GCRuntimeException;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.google.common.base.Joiner;
import com.google.common.base.Splitter;
import com.google.common.collect.Maps;
import org.apache.commons.lang.StringUtils;

import java.io.Serializable;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;

/**
 * Created by liuyao on 2017/1/11. 商品多档位解析类
 */
public class PriceDegreeDto implements Serializable {
	private static final long serialVersionUID = 3281272373039598785L;

	public static final String BIZPARAMS_REX = "ψ";
	public static final String PRICE_KEY = "price";
	public static final String CREDITS_KEY = "credits";
	public static final String TITLE_KEY = "title";
	public static final String LIMITCOUNT_KEY = "limitCount";
	public static final String LIMITSCOPE_KEY = "limitScope";

	private static Splitter spl = Splitter.on(BIZPARAMS_REX).trimResults().omitEmptyStrings();
	private static Joiner join = Joiner.on(BIZPARAMS_REX).skipNulls();

	/**
	 * 档位->(credits->积分，title->文案)
	 */
	private TreeMap<String, Map<String, String>> customDegree;

	/**
	 * Comparator<String>
	 */
	private static Comparator<String> comparator = new Comparator<String>() {
		@Override
		public int compare(String o1, String o2) {
			try {
				Long o1l = Long.valueOf(o1);
				Long o2l = Long.valueOf(o2);
				return o1l.compareTo(o2l);
			} catch (NumberFormatException e) {
				return o1.compareTo(o2);
			}
		}
	};

	/**
	 * 默认构造函数
	 */
	public PriceDegreeDto() {
	}

	/**
	 * 传入多档位信息
	 * 
	 * @param cps
	 */
	public PriceDegreeDto(String[] cps) {
		TreeMap<String, Map<String, String>> map = Maps.newTreeMap(comparator);
		for (String cp : cps) {
			Map<String, String> hashmap = new HashMap<>();
			if (cp.split(":").length > 2) {
				hashmap.put(CREDITS_KEY, cp.split(":")[1]);
				hashmap.put(TITLE_KEY, cp.split(":")[2]);
				map.put(cp.split(":")[0], hashmap);
			} else if (cp.split(":").length == 2) {
				hashmap.put(CREDITS_KEY, cp.split(":")[1]);
				map.put(cp.split(":")[0], hashmap);
			} else {
				hashmap.put(CREDITS_KEY, null);
				map.put(cp.split(":")[0], hashmap);
			}
		}
		this.customDegree = map;
	}

	/**
	 * 反序列化构造方法
	 * 
	 * @param customDegreeStr
	 */
	public PriceDegreeDto(String customDegreeStr) {
		if(StringUtils.isBlank(customDegreeStr)){
			this.customDegree = Maps.newTreeMap();
			return;
		}
		TreeMap<String, Map<String, String>> map = Maps.newTreeMap(comparator);
		JSONArray ja = JSON.parseArray(customDegreeStr);
		for (int i = 0; i < ja.size(); i++) {
			Map<String, String> hashmap = Maps.newHashMap();
			if (ja.getJSONObject(i).getString(LIMITCOUNT_KEY) != null) {
				hashmap.put(CREDITS_KEY, getDegreeCredits(ja.getJSONObject(i).getString(CREDITS_KEY)));
				hashmap.put(LIMITCOUNT_KEY, ja.getJSONObject(i).getString(LIMITCOUNT_KEY));
				hashmap.put(LIMITSCOPE_KEY, ja.getJSONObject(i).getString(LIMITSCOPE_KEY));
				map.put(ja.getJSONObject(i).getString(PRICE_KEY), hashmap);
			} else if (ja.getJSONObject(i).getString(TITLE_KEY) != null) {
				hashmap.put(CREDITS_KEY, getDegreeCredits(ja.getJSONObject(i).getString(CREDITS_KEY)));
				hashmap.put(TITLE_KEY, ja.getJSONObject(i).getString(TITLE_KEY));
				map.put(ja.getJSONObject(i).getString(PRICE_KEY), hashmap);
			} else {
				hashmap.put(CREDITS_KEY, getDegreeCredits(ja.getJSONObject(i).getString(CREDITS_KEY)));
				map.put(ja.getJSONObject(i).getString(PRICE_KEY), hashmap);
			}
		}
		this.customDegree = map;
	}

	/**
	 * 获取虚拟商品订单中的档位
	 * 
	 * @param bizParams
	 * @return
	 */
	public static String getDegree(String bizParams) {
		if (StringUtils.isBlank(bizParams)) {
			return "";
		}
		List<String> paramlist = spl.splitToList(bizParams);
		return paramlist.get(0);
	}

	/**
	 * 获取虚拟商品订单中的用户账号
	 * 
	 * @param bizParams
	 * @return
	 */
	public static String getAccount(String bizParams) {
		if (StringUtils.isBlank(bizParams)) {
			return "";
		}
		List<String> paramlist = spl.splitToList(bizParams);
		if (paramlist.size() >= 2) {
			return paramlist.get(1);
		} else {
			return "";
		}
	}

	/**
	 * 构建虚拟商品订单的bizParams
	 * 
	 * @param degree
	 * @param account
	 * @return
	 */
	public static String getBizParams(String degree, String account) {
		return join.join(degree, account);
	}

	/**
	 * getDegreeCredits
	 * 
	 * @param creditsStr
	 * @return
	 */
	private String getDegreeCredits(String creditsStr) {
		if (creditsStr == null || StringUtils.isBlank(creditsStr.trim())) {
			return null;
		} else {
			return creditsStr.trim();
		}
	}

	/**
	 * 判断面额是否存在
	 * 
	 * @param degree
	 * @return
	 */
	public boolean isDegreeExist(String degree) {
		return customDegree.containsKey(degree);
	}

	/**
	 * 序列化为json格式的String，用于保存。
	 * 
	 * @return
	 */
	public String serializePriceDegree() {
		JSONArray ja = new JSONArray();
		for (Map.Entry<String, Map<String, String>> entry : customDegree.entrySet()) {
			JSONObject jo = new JSONObject();
			jo.put(PRICE_KEY, entry.getKey());
			jo.put(CREDITS_KEY, entry.getValue().get(CREDITS_KEY));
			jo.put(TITLE_KEY, entry.getValue().get(TITLE_KEY));
			jo.put(LIMITCOUNT_KEY, entry.getValue().get(LIMITCOUNT_KEY));
			jo.put(LIMITSCOPE_KEY, entry.getValue().get(LIMITSCOPE_KEY));
			ja.add(jo);
		}
		return ja.toString();
	}

	/**
	 * serializeUp200Alipay
	 */
	public void serializeUp200Alipay() {
		for (Map.Entry<String, Map<String, String>> entry : customDegree.entrySet()) {
			if (Integer.parseInt(entry.getKey()) > 200) {
				throw new GCRuntimeException("支付宝面值不能大于200");
			}
		}
	}

	/**
	 * serializeUp200Alipay
	 */
	public void serializeUp500Alipay() {
		for (Map.Entry<String, Map<String, String>> entry : customDegree.entrySet()) {
			if (Integer.parseInt(entry.getKey()) > 500) {
				throw new GCRuntimeException("支付宝面值不能大于500");
			}
		}
	}

	/**
	 * 根据档位获取积分
	 * 
	 * @param degree
	 * @return
	 */
	public String getCreditsByDegree(String degree) {
		return customDegree.get(degree).get(CREDITS_KEY);
	}

	/**
	 * 根据档位获取文案
	 * 
	 * @param degree
	 * @return
	 */
	public String getTitleByDegree(String degree) {
		return customDegree.get(degree).get(TITLE_KEY);
	}

	/**
	 * 根据档位获取 限制数和限制类型
	 * 
	 * @param degree
	 * @return
	 */
	public Map<String, String> getLimitByDegree(String degree) {
		return customDegree.get(degree);
	}

	/**
	 * 取出最小的档位key
	 * 
	 * @return
	 */
	public String getMinDegreeKey() {
		return customDegree.firstKey();
	}

	/**
	 * 取出最小的档位key
	 * 
	 * @return
	 */
	public Integer getMinDegreeInt() {
		return Integer.valueOf(getMinDegreeKey());
	}

	/**
	 * 取出最小的档位value的积分
	 * 
	 * @return
	 */
	public String getMinDegreeCredits() {
		String minDegreeCredits = customDegree.firstEntry().getValue().get(CREDITS_KEY);
		if (StringUtils.isBlank(minDegreeCredits)) {
			return null;
		}

		String temp = "";
		for (Map.Entry<String, Map<String, String>> entry : customDegree.entrySet()){
			Map<String, String> ent = entry.getValue();
			if (StringUtils.isBlank(temp)) {
				temp = ent.get(CREDITS_KEY);
			} else {
				if (StringUtils.isNotBlank(ent.get(CREDITS_KEY)) && Long.valueOf(temp).compareTo(Long.valueOf(ent.get(CREDITS_KEY))) > 0) {
					temp = ent.get(CREDITS_KEY);
				}
			}
		}

		if (StringUtils.isBlank(temp)) {
			return minDegreeCredits;
		}

		return temp;
	}

	/**
	 * CreditsComparator
	 */
	static class CreditsComparator implements Comparator<Object>, Serializable {
		private static final long serialVersionUID = -2414083834670521985L;

		@Override
		public int compare(Object object1, Object object2) {// 实现接口中的方法
			String p1 = (String) object1; // 强制转换
			String p2 = (String) object2;
			return p2.compareTo(p1);
		}
	}

	/**
	 * 取出最小的档位value的文案
	 * 
	 * @return
	 */
	public String getMinDegreeTitle() {
		return customDegree.firstEntry().getValue().get(TITLE_KEY);
	}

	/**
	 * 返回商品的多档位信息
	 * 
	 * @return
	 */
	public TreeMap<String, Map<String, String>> getCustomDegreeMap() {
		return customDegree;
	}

	/**
	 * 取出最小的积分
	 * 
	 * @param creditsRate
	 *            app.creditsRate
	 * @return
	 */
	public Long getMinCredits(Integer creditsRate) {
		Long a = 0L;
		for (Map.Entry<String, Map<String, String>> entry : customDegree.entrySet()){
			if (a == 0L) {
				a = getCreditsValue(Long.valueOf(creditsRate), entry);
			} else {
				a = getMinCreditsValue(Long.valueOf(creditsRate), a, entry);
			}
		}

		return a;
	}
	
	   /**
     * 取出最小的积分
     * 
     * @param creditsRate
     *            app.creditsRate
     * @return
     */
    public Long getMinCredits(Long creditsRate) {
        Long a = 0L;
		for (Map.Entry<String, Map<String, String>> entry : customDegree.entrySet()){
			if (a == 0L) {
				a = getCreditsValue(creditsRate, entry);
			} else {
				a = getMinCreditsValue(creditsRate, a, entry);
			}
		}
        return a;
    }

	private Long getMinCreditsValue(Long creditsRate, Long minValue, Map.Entry<String, Map<String, String>> entry) {
        Long result = minValue;
		if (entry.getValue().get(CREDITS_KEY) != null) {
            if (minValue.compareTo(Long.valueOf(entry.getValue().get(CREDITS_KEY))) > 0) {
                result = Long.valueOf(entry.getValue().get(CREDITS_KEY));
            }
        } else {
            if (minValue.compareTo(Long.valueOf(entry.getKey()) * creditsRate) > 0) {
                result = Long.valueOf(entry.getKey()) * creditsRate;
            }
        }
		return result;
	}

	private Long getCreditsValue(Long creditsRate, Map.Entry<String, Map<String, String>> entry) {
		if (entry.getValue().get(CREDITS_KEY) != null) {
            return Long.valueOf(entry.getValue().get(CREDITS_KEY));
        } else {
            return Long.valueOf(entry.getKey()) * creditsRate;
        }
	}

	/**
	 * 判断是否单一档位
	 * 
	 * @return
	 */
	public boolean isSingleDegree() {
		if (customDegree.size() == 1) {
			return true;
		}
		return false;
	}

}
