package cn.com.duiba.goods.common.configuration;

import cn.com.duiba.wolf.entity.Pair;
import cn.com.duibaboot.ext.autoconfigure.core.SpecifiedBeanPostProcessor;
import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.Multimap;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.beans.BeansException;
import org.springframework.web.method.HandlerMethod;
import org.springframework.web.servlet.mvc.method.RequestMappingInfo;
import org.springframework.web.servlet.mvc.method.annotation.RequestMappingHandlerMapping;

import javax.annotation.Resource;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * 重新注册 RequestMappingInfo，给控制器 path 加上前缀
 * 如果存在多个 PrefixHandler 的匹配包路径存在交集，那么交集中的控制器方法会被重复注册
 * 比如 /api1 和 /api2 两个前缀都匹配到同一个包下的某个控制器方法 /aaa/bbb
 * 那么本类运行完成后 /aaa/bbb path将不再存在，取而代之的是 /api1/aaa/bbb 以及 /api2/aaa/bbb ，两者指向同一个方法
 * Created by guoyanfei .
 * 2022/11/14 .
 */
public class ControllerPrefixBeanPostProcessor implements SpecifiedBeanPostProcessor<RequestMappingHandlerMapping> {

    /**
     * 前缀控制器
     */
    @Resource
    private Set<PrefixHandler> prefixHandlers;

    @Override
    public Class<RequestMappingHandlerMapping> getBeanType() {
        return RequestMappingHandlerMapping.class;
    }

    @Override
    public Object postProcessBeforeInitialization(RequestMappingHandlerMapping bean, String beanName) throws BeansException {
        return bean;
    }

    @Override
    public Object postProcessAfterInitialization(RequestMappingHandlerMapping bean, String beanName) throws BeansException {
        if (CollectionUtils.isEmpty(prefixHandlers)) {
            return bean;
        }
        Map<RequestMappingInfo, HandlerMethod> methodMap = bean.getHandlerMethods();

        Multimap<PrefixHandler, Pair<RequestMappingInfo, HandlerMethod>> reregisterMap = ArrayListMultimap.create();

        // 过滤出需要增加前缀的所有控制器 RequestMappingInfo，准备重新注册
        for (Map.Entry<RequestMappingInfo, HandlerMethod> m : methodMap.entrySet()) {
            RequestMappingInfo info = m.getKey();
            HandlerMethod handlerMethod = m.getValue();
            for (PrefixHandler handler : prefixHandlers) {
                // 匹配该控制器类所在的包 和 handler 需要匹配的包 是否匹配上
                if (!handler.match(handlerMethod.getBeanType())) {
                    continue;
                }
                // 匹配上，加入需要重新注册的map
                reregisterMap.put(handler, Pair.from(info, handlerMethod));
            }
        }

        // 没有需要重新注册的，直接走
        if (reregisterMap.isEmpty()) {
            return bean;
        }

        // 重新注册
        for (PrefixHandler handler : reregisterMap.keySet()) {
            List<Pair<RequestMappingInfo, HandlerMethod>> pairs = (List<Pair<RequestMappingInfo, HandlerMethod>>) reregisterMap.get(handler);
            for (Pair<RequestMappingInfo, HandlerMethod> it : pairs) {
                RequestMappingInfo info = it.getKey();
                HandlerMethod handlerMethod = it.getValue();
                // 取消注册老的info
                bean.unregisterMapping(info);
                // 根据老的info拼接一个新的info，重新注册
                bean.registerMapping(handler.combineRequestMappingInfo(info), handlerMethod.getBean(), handlerMethod.getMethod());
            }
        }
        return bean;
    }

    @Override
    public int getOrder() {
        return 0;
    }

}
