package cn.com.duiba.goods.common.utils;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.*;

/**
 * 批量计算工具
 * Created by guoyanfei .
 * 2022/12/27 .
 */
public final class BatchEngine<T, R> {

    /**
     * 执行批量任务，拿结果
     *
     * @param todoList
     * @param perBatchSize
     * @param executorService
     * @param callable
     * @return
     */
    public static <T, R> List<R> submit(List<T> todoList, int perBatchSize, ExecutorService executorService, BatchCallable<T, R> callable) {
        int pageCount = PagerUtils.pageCount(perBatchSize, todoList.size());
        List<R> rList = new ArrayList<>(todoList.size());
        for (int pageIndex = 0; pageIndex < pageCount; pageIndex++) {
            int fromIndex = pageIndex * perBatchSize;
            int toIndex = Math.min((pageIndex + 1) * perBatchSize, todoList.size());
            List<T> temp = todoList.subList(fromIndex, toIndex);
            List<Future<R>> futureList = new ArrayList<>(temp.size());
            for (T it : temp) {
                Future<R> future = executorService.submit(new BatchPerTask<T, R>(it) {

                    @Override
                    public R doCall(T data) {
                        return callable.call(data);
                    }
                });
                futureList.add(future);
            }
            try {
                for (Future<R> future : futureList) {
                    rList.add(future.get());
                }
            } catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            } catch (ExecutionException e) {
                throw new RuntimeException(e);
            }
            if (Thread.currentThread().isInterrupted()) {
                break;
            }
        }
        return rList;
    }

    /**
     * 执行批量任务，无结果
     *
     * @param todoList
     * @param perBatchSize
     * @param executorService
     * @param runnable
     */
    public static <T> void execute(List<T> todoList, int perBatchSize, ExecutorService executorService, BatchCallable<T, Void> callable) {
        int pageCount = PagerUtils.pageCount(perBatchSize, todoList.size());
        for (int pageIndex = 0; pageIndex < pageCount; pageIndex++) {
            int fromIndex = pageIndex * perBatchSize;
            int toIndex = Math.min((pageIndex + 1) * perBatchSize, todoList.size());
            List<T> temp = todoList.subList(fromIndex, toIndex);
            List<Future<Void>> futureList = new ArrayList<>(temp.size());
            for (T it : temp) {
                Future<Void> future = executorService.submit(new BatchPerTask<T, Void>(it) {

                    @Override
                    public Void doCall(T data) {
                        return callable.call(data);
                    }
                });
                futureList.add(future);
            }
            try {
                for (Future<Void> future : futureList) {
                    future.get();
                }
            } catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            } catch (ExecutionException e) {
                throw new RuntimeException(e);
            }
            if (Thread.currentThread().isInterrupted()) {
                break;
            }
        }
    }

}
