package cn.com.duiba.cloud.jiuli.client.service;

import cn.com.duiba.boot.exception.BizException;
import cn.com.duiba.cloud.jiuli.client.JiuliFileClient;
import cn.com.duiba.cloud.jiuli.client.configuration.FileSpaceProperties;
import cn.com.duiba.cloud.jiuli.client.domian.dto.FileDto;
import cn.com.duiba.cloud.jiuli.client.domian.params.*;
import cn.com.duiba.cloud.jiuli.client.domian.request.PutFileRequest;
import cn.com.duiba.cloud.jiuli.client.domian.result.CdnFlushResult;
import cn.com.duiba.cloud.jiuli.client.domian.result.GetFileResult;
import cn.com.duiba.cloud.jiuli.client.domian.result.PutFileResult;
import cn.com.duiba.cloud.jiuli.client.remote.RemoteJiuliCdnApi;
import cn.com.duiba.cloud.jiuli.client.remote.RemoteJiuliFileApi;
import cn.com.duiba.cloud.jiuli.client.remote.RemoteJiuliImageApi;
import cn.com.duiba.cloud.jiuli.client.service.auth.JiuliClientAuthRequestInterceptor;
import cn.com.duiba.cloud.jiuli.client.utils.MultipartFileUtils;
import com.google.common.base.Joiner;
import feign.Response;
import lombok.extern.slf4j.Slf4j;
import org.springframework.web.multipart.MultipartFile;

import javax.annotation.Resource;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Objects;

/**
 * @author liuyao
 */
@Slf4j
public class JiuliFileClientImpl implements JiuliFileClient {

    /**
     * 空间标示
     */
    private String spaceKey;

    private FileSpaceProperties fileSpaceProperties;

    @Resource
    private RemoteJiuliFileApi remoteJiuliFileClient;
    @Resource
    private RemoteJiuliImageApi remoteJiuliImageApi;
    @Resource
    private RemoteJiuliCdnApi remoteJiuliCdnApi;
    @Resource
    private JiuliClientAuthRequestInterceptor jiuliClientAuthRequestInterceptor;

    public void init(){
        Objects.requireNonNull(spaceKey);
        Objects.requireNonNull(fileSpaceProperties);
        jiuliClientAuthRequestInterceptor.registerSecret(spaceKey,fileSpaceProperties.getSecret());
    }

    @Override
    public String getSpaceKey() {
        return this.spaceKey;
    }

    @Override
    public List<FileDto> listFile(ListFileParams params) throws BizException {
        return remoteJiuliFileClient.listFile(spaceKey, params);
    }

    @Override
    public void createFolder(String filePath) throws BizException {
        CreateDirParams param = new CreateDirParams(filePath);
        remoteJiuliFileClient.createFolder(spaceKey, param);
    }

    @Override
    public PutFileResult putFile(String fileId, PutFileRequest request) throws BizException {

        Objects.requireNonNull(fileId,"fileId 不能为空");

        PutFileParams param = new PutFileParams();
        param.setFileId(fileId);
        param.setMetadata(request.getMetadata());
        if(Objects.isNull(request.getFile())){
            throw new IllegalArgumentException("缺少上传内容");
        }
        try{
            return remoteJiuliFileClient.putFile(spaceKey, param, request.getFile());
        }finally {
            MultipartFileUtils.cleanLocalFile(request.getFile());
        }
    }

    @Override
    public GetFileResult getFile(String filePath) throws BizException {
        try {
            Joiner joiner = Joiner.on(",").skipNulls();
            Response response = remoteJiuliFileClient.fetchFile(spaceKey, filePath);
            Map<String, Collection<String>> headers = response.headers();
            GetFileResult result = new GetFileResult();
            result.setFileContext(response.body().asInputStream());
            for(String key:headers.keySet()){
                result.addHeader(key,joiner.join(headers.get(key)));
            }
            return result;
        }catch (IOException e){
            log.error("获取文件流失败",e);
            throw new BizException("获取文件流失败");
        }
    }

    @Override
    public void getFile(String filePath, File file) throws BizException, IOException {
        try (
                InputStream inputStream = getFile(filePath).getFileContext();
                FileOutputStream outputStream = new FileOutputStream(file)
        ){
            int read;
            byte[] bytes = new byte[1024];
            while ((read = inputStream.read(bytes)) != -1) {
                outputStream.write(bytes, 0, read);
            }
        }
    }

    @Override
    public Boolean fileExist(String fileId) throws BizException {
        return remoteJiuliFileClient.fileExist(spaceKey,fileId).getExist();
    }

    @Override
    public void deleteFile(String fileId) throws BizException {
        remoteJiuliFileClient.deleteFile(spaceKey, fileId);
    }

    @Override
    public void appendToFile(String fileId, Long position, InputStream inputStream) throws BizException {
        AppendFileParams params = new AppendFileParams();
        params.setFileId(fileId);
        params.setPosition(position);
        MultipartFile multipartFile = MultipartFileUtils.fileToMultipartFile(inputStream);
        remoteJiuliFileClient.appendFile(spaceKey, params,multipartFile);
    }

    @Override
    public String imageChange(ImageChangeParams params) throws BizException {
        return remoteJiuliImageApi.imageChange(spaceKey,params);
    }

    @Override
    public List<CdnFlushResult> refreshCdn(RefreshCdnParams params) throws BizException {
        return remoteJiuliCdnApi.refreshCdn(spaceKey,params);
    }

    public void setSpaceKey(String spaceKey) {
        this.spaceKey = spaceKey;
    }

    public void setFileSpaceProperties(FileSpaceProperties fileSpaceProperties) {
        this.fileSpaceProperties = fileSpaceProperties;
    }

}
