package cn.com.duiba.cloud.jiuli.client.service.auth;

import cn.com.duiba.cloud.jiuli.client.domian.constants.HttpHeaders;
import cn.com.duiba.cloud.jiuli.client.domian.constants.RequestParamConstants;
import cn.com.duiba.cloud.jiuli.client.domian.dto.AuthDto;
import cn.com.duiba.cloud.jiuli.client.domian.params.AuthParams;
import cn.com.duiba.cloud.jiuli.client.remote.RemoteJiuliAuthApi;
import com.github.benmanes.caffeine.cache.CacheLoader;
import com.github.benmanes.caffeine.cache.Caffeine;
import com.github.benmanes.caffeine.cache.LoadingCache;
import com.google.common.collect.Maps;
import feign.RequestInterceptor;
import feign.RequestTemplate;
import org.apache.commons.lang3.StringUtils;
import org.aspectj.lang.annotation.Aspect;
import org.springframework.beans.factory.annotation.Value;

import javax.annotation.Resource;
import java.util.Collection;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.TimeUnit;

/**
 * @author liuyao
 */
@Aspect
public class JiuliClientAuthRequestInterceptor implements RequestInterceptor {

    @Resource
    private RemoteJiuliAuthApi remoteJiuliAuthApi;
    @Value("${spring.application.name}")
    private String appId;

    /**
     * 提前5分钟失效缓存
     */
    private static final long OUTMODED_TIME = 5*60*1000L;

    private final Map<String,String> secretMap = Maps.newConcurrentMap();

    public void registerSecret(String spaceKey, String secret) {
        secretMap.put(spaceKey,secret);
    }

    /**
     *  AuthDto 有效期是2小时，缓存提前5分钟失效
     */
    private final LoadingCache<String,AuthDto> stateCache = Caffeine.newBuilder().expireAfterWrite(115, TimeUnit.MINUTES).build(new CacheLoader<String, AuthDto>() {
        @Override
        public AuthDto load(String spaceKey) throws Exception {
            String secret = secretMap.get(spaceKey);
            if(Objects.isNull(secret)){
                throw new RuntimeException("未设置文件空间["+spaceKey+"]的秘钥，无法对该空间进行操作");
            }
            long timestamp = System.currentTimeMillis();
            String signature = SignatureUtils.sign(spaceKey,secret,appId,timestamp);

            AuthParams params = new AuthParams();
            params.setAppId(appId);
            params.setSpaceKey(spaceKey);
            params.setTimestamp(timestamp);
            params.setSignature(signature);
            return remoteJiuliAuthApi.getState(params);
        }
    });

    @Override
    public void apply(RequestTemplate template) {
        String spaceKey = getSpaceKey(template);
        if(StringUtils.isBlank(spaceKey)){
            return;
        }
        AuthDto authDto = stateCache.get(spaceKey);
        Objects.requireNonNull(authDto);
        template.header(HttpHeaders.AUTHORIZATION, authDto.getState());
    }

    private String getSpaceKey(RequestTemplate template){
        Map<String, Collection<String>> queries = template.queries();
        if(queries.containsKey(RequestParamConstants.SPACE_KEY)){
            return queries.get(RequestParamConstants.SPACE_KEY).toArray(new String[0])[0];
        }
        return null;
    }

}
