package cn.com.duiba.cloud.jiuli.client.domian.params;

import cn.com.duiba.cloud.jiuli.client.domian.constants.HttpHeaders;
import cn.com.duiba.cloud.jiuli.client.utils.DateUtil;
import com.google.common.collect.Maps;

import java.io.Serializable;
import java.text.ParseException;
import java.util.Collections;
import java.util.Date;
import java.util.Map;

/**
 * @author liuyao
 */
public class MetadataInfo implements Serializable {

    private final Map<String,String> metadata = Maps.newHashMap();

    public void setHeader(String key, String value) {
        metadata.put(HttpHeaders.formatHeader(key), value);
    }

    /**
     * Removes the http header (SDK internal usage only).
     *
     * @param key
     *            The key of header.
     */
    public void removeHeader(String key) {
        metadata.remove(HttpHeaders.formatHeader(key));
    }


    /**
     * Gets the value of Last-Modified header, which means the last modified
     * time of the object.
     *
     * @return Object's last modified time.
     */
    public Date getLastModified() {
        if(!metadata.containsKey(HttpHeaders.LAST_MODIFIED)){
            return null;
        }
        try{
            return DateUtil.parseRfc822Date(metadata.get(HttpHeaders.LAST_MODIFIED));
        }catch (ParseException e){
            return null;
        }
    }

    /**
     * Sets the value of Last-Modified header, which means the last modified
     * time of the object.
     *
     * @param lastModified
     *            Object's last modified time.
     */
    public void setLastModified(Date lastModified) {
        metadata.put(HttpHeaders.LAST_MODIFIED, DateUtil.formatRfc822Date(lastModified));
    }

    /**
     * Gets the {@link Date} value of the "Expires" header in Rfc822 format. If
     * expiration is not set, then the value is null.
     *
     * @return Expires header's value in Rfc822.
     * @throws ParseException
     *             The value is not in the Rfc822 format.
     */
    public Date getExpirationTime() throws ParseException {
        String expires = metadata.get(HttpHeaders.EXPIRES);
        if (expires != null){
            return DateUtil.parseRfc822Date(metadata.get(HttpHeaders.EXPIRES));
        }
        return null;
    }

    /**
     * Gets the string value of the "Expires" header in Rfc822 format. If
     * expiration is not set, then the value is null.
     *
     * @return The string value of "Expires" header.
     */
    public String getRawExpiresValue() {
        return metadata.get(HttpHeaders.EXPIRES);
    }

    /**
     * Sets the "Expires" header.
     *
     * @param expirationTime
     *            Expiration time.
     */
    public void setExpirationTime(Date expirationTime) {
        metadata.put(HttpHeaders.EXPIRES, DateUtil.formatRfc822Date(expirationTime));
    }

    /**
     * Gets Content-Length header, which is the object content's size.
     *
     * @return The object content's size.
     */
    public Long getContentLength() {
        String contentLength = metadata.get(HttpHeaders.CONTENT_LENGTH);
        return contentLength == null ? 0L : Long.parseLong(contentLength);
    }

    /**
     * Sets the Content-Length header to indicate the object's size. The correct
     * Content-Length header is needed for a file upload.
     *
     * @param contentLength
     *            Object content size.
     */
    public void setContentLength(Long contentLength) {
        metadata.put(HttpHeaders.CONTENT_LENGTH,contentLength.toString());
    }

    /**
     * Gets the Content-Type header to indicate the object content's type in
     * MIME type format.
     *
     * @return The content-type header in MIME type format.
     */
    public String getContentType() {
        return metadata.get(HttpHeaders.CONTENT_TYPE);
    }

    /**
     * Sets the Content-Type header to indicate the object content's type in
     * MIME type format.
     *
     * @param contentType
     * The content-type header in MIME type format.
     */
    public void setContentType(String contentType) {
        metadata.put(HttpHeaders.CONTENT_TYPE, contentType);
    }

    public String getContentMd5() {
        return metadata.get(HttpHeaders.CONTENT_MD5);
    }

    public void setContentMd5(String md5){
        metadata.put(HttpHeaders.CONTENT_MD5,md5);
    }

    /**
     * Gets the Content-Encoding header which is to encode the object content.
     *
     * @return Object content's encoding.
     */
    public String getContentEncoding() {
        return metadata.get(HttpHeaders.CONTENT_ENCODING);
    }

    /**
     * Sets the Content-Encoding header which is to encode the object content.
     *
     * @param encoding
     *            Object content's encoding.
     */
    public void setContentEncoding(String encoding) {
        metadata.put(HttpHeaders.CONTENT_ENCODING, encoding);
    }

    /**
     * Gets the Cache-Control header. This is the standard http header.
     *
     * @return Cache-Control header.
     */
    public String getCacheControl() {
        return metadata.get(HttpHeaders.CACHE_CONTROL);
    }

    /**
     * Sets the Cache-Control header. This is the standard http header.
     *
     * @param cacheControl
     *            Cache-Control header.
     */
    public void setCacheControl(String cacheControl) {
        metadata.put(HttpHeaders.CACHE_CONTROL, cacheControl);
    }

    /**
     * Gets the Content-Disposition header.This is the standard http header.
     *
     * @return Content-Disposition header.
     */
    public String getContentDisposition() {
        return metadata.get(HttpHeaders.CONTENT_DISPOSITION);
    }

    /**
     * Sets Content-Disposition header.
     *
     * @param disposition
     *            Content-Disposition header.
     */
    public void setContentDisposition(String disposition) {
        metadata.put(HttpHeaders.CONTENT_DISPOSITION, disposition);
    }

    /**
     * Gets the raw metadata (SDK internal usage only). The value returned is
     * immutable.
     *
     * @return The raw metadata object.
     */
    public Map<String, String> getRawMetadata() {
        return Collections.unmodifiableMap(metadata);
    }

}
