package cn.com.duiba.cloud.jiuli.client.service;

import cn.com.duiba.boot.exception.BizException;
import cn.com.duiba.cloud.jiuli.client.JiuliFileClient;
import cn.com.duiba.cloud.jiuli.client.domian.constants.HttpHeaders;
import cn.com.duiba.cloud.jiuli.client.domian.dto.FileDto;
import cn.com.duiba.cloud.jiuli.client.domian.params.*;
import cn.com.duiba.cloud.jiuli.client.domian.result.AppendResult;
import cn.com.duiba.cloud.jiuli.client.domian.result.DeleteResult;
import cn.com.duiba.cloud.jiuli.client.domian.result.FetchResult;
import cn.com.duiba.cloud.jiuli.client.domian.result.PutFileResult;
import cn.com.duiba.cloud.jiuli.client.remote.RemoteJiuliFileApi;
import cn.com.duiba.cloud.jiuli.client.service.auth.JiuliClientAuthRequestInterceptor;
import cn.com.duiba.cloud.jiuli.client.utils.MultipartFileUtils;
import com.google.common.base.Joiner;
import feign.Response;
import lombok.extern.slf4j.Slf4j;
import org.springframework.web.multipart.MultipartFile;

import javax.annotation.Resource;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.Collection;
import java.util.List;
import java.util.Map;

/**
 * @author liuyao
 */
@Slf4j
public class JiuliFileClientImpl implements JiuliFileClient {

    private static final Joiner HEADER_JOINER = Joiner.on(",").skipNulls();
    /**
     * 空间标示
     */
    private String spaceKey;

    @Resource
    private RemoteJiuliFileApi remoteJiuliFileClient;

    @Override
    public String getSpaceKey() {
        return this.spaceKey;
    }

    @Override
    public List<FileDto> findChildFiles(ListFileParams params) throws BizException {
        return remoteJiuliFileClient.listFile(spaceKey, params);
    }

    @Override
    public FetchResult fetchFile(String fileId) throws BizException {
        Response response = remoteJiuliFileClient.fetchFile(spaceKey, fileId);
        Map<String, Collection<String>> headers = response.headers();
        FetchResult result = new FetchResult();
        try {
            MetadataInfo metadataInfo = new MetadataInfo();
            for(String key:headers.keySet()){
                metadataInfo.setHeader(key,HEADER_JOINER.join(headers.get(key)));
            }
            result.setMetadataInfo(metadataInfo);
            result.setInputStream(response.body().asInputStream());
        }catch (IOException e){
            throw new BizException("获取文件流失败:"+e.getMessage());
        }
        return result;
    }


    @Override
    public void createFolder(String filePath) throws BizException {
        CreateDirParams param = new CreateDirParams(filePath);
        remoteJiuliFileClient.createFolder(spaceKey, param);
    }

    @Override
    public PutFileResult putFile(PutFileParams params, File file) throws BizException {
        MultipartFile multipartFile = MultipartFileUtils.fileToMultipartFile(file);
        try{
            return remoteJiuliFileClient.putFile(spaceKey, params, multipartFile);
        }finally {
            MultipartFileUtils.cleanLocalFile(multipartFile);
        }
    }

    @Override
    public PutFileResult putFile(PutFileParams params, InputStream inputStream) throws BizException {
        MultipartFile multipartFile = MultipartFileUtils.fileToMultipartFile(inputStream);
        try{
            return remoteJiuliFileClient.putFile(spaceKey, params, multipartFile);
        }finally {
            MultipartFileUtils.cleanLocalFile(multipartFile);
        }
    }

    @Override
    public PutFileResult putFile(PutFileParams params, MultipartFile multipartFile) throws BizException {
        try{
            return remoteJiuliFileClient.putFile(spaceKey, params, multipartFile);
        }finally {
            MultipartFileUtils.cleanLocalFile(multipartFile);
        }
    }

    @Override
    public AppendResult appendToFile(AppendParams params, InputStream inputStream) throws BizException {
        MultipartFile multipartFile = MultipartFileUtils.fileToMultipartFile(inputStream);
        try{
            return remoteJiuliFileClient.appendFile(spaceKey, params,multipartFile);
        }finally {
            MultipartFileUtils.cleanLocalFile(multipartFile);
        }
    }

    @Override
    public Boolean fileExist(String fileId) throws BizException {
        return remoteJiuliFileClient.fileExist(spaceKey,fileId).getExist();
    }

    @Override
    public DeleteResult deleteFile(String fileId) throws BizException {
        return remoteJiuliFileClient.deleteFile(spaceKey, fileId);
    }

    public void setSpaceKey(String spaceKey) {
        this.spaceKey = spaceKey;
    }

}
