package cn.com.duiba.kjy.base.api.utils;

import lombok.extern.slf4j.Slf4j;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Random;

/**
 * 红包随机算法工具类
 * @author lizhi
 * @date 2020/4/2 2:24 PM
 */
@Slf4j
public class RedPacketRandomUtils {

    private static Random random = new Random();

    public static void main(String[] args) {
        long start = System.currentTimeMillis();
        int redPacketAmount = 160;
        int redPacketNum = 3;
        int minAmount = 30;
        List<Integer> amountList = randomAlgorithm1(redPacketNum, redPacketAmount, minAmount);
        for (Integer amount : amountList) {
            System.out.println(amount);
        }
        System.out.println("======");
        System.out.println(amountList.stream().mapToInt(amount -> amount).sum());
        //平均值
        double ave = redPacketAmount * 1.0 / redPacketNum;
        //单个红包最大金额，平均值的2倍
        int max = (int) ave * 2;
        System.out.println(ave);
        System.out.println(max);
        System.out.println("===");
        System.out.println(System.currentTimeMillis() - start);
    }

    /**
     * 随机算法1
     * 注意，若 金额 < (个数 * 单个最小金额)，直接返回空集合
     * 若创建失败，也会返回空集合（初期实验阶段，可能会出现）
     * @param redPacketNum 红包个数
     * @param redPacketAmount 红包金额，单位分
     * @param minAmount 单个红包最小金额，单位分
     * @return 拆分好的金额数组，单位分
     */
    public static List<Integer> randomAlgorithm1(int redPacketNum, int redPacketAmount, int minAmount) {
        if (redPacketNum * minAmount > redPacketAmount) {
            return Collections.emptyList();
        }
        List<Integer> result = new ArrayList<>();
        //平均值
        double ave = redPacketAmount * 1.0 / redPacketNum;
        //单个红包最大金额，平均值的2倍
        int max = (int) ave * 2;
        //偏移量
        double offset = 0;
        int originalAmount = redPacketAmount;
        int originalNum = redPacketNum;
        int totalAmount = 0;
        while (redPacketNum > 0) {
            if (redPacketNum == 1) {
                //最后一个，剩余金额都归它
                result.add(redPacketAmount);
                totalAmount += redPacketAmount;
                if (redPacketAmount > max || redPacketAmount < minAmount) {
                    //理论上不会出现，最后的保障
                    log.error("randomAlgorithm1, amount error, redPacketNum={}, redPacketAmount={}, minAmount={}, amount={}", originalNum, originalAmount, minAmount, redPacketAmount);
                    return Collections.emptyList();
                }
                break;
            }
            //单个红包区间：偏移量小于0时，[平均值 - 最大值]，
            //大于等于0时，[最小值 - 平均值]
            int singleMin = offset < 0 ? (int) ave : minAmount;
            int singleMax = offset < 0 ? max : (int) ave;
            int amount = random.nextInt(singleMax);
            if (amount <= singleMin) {
                //小于最小金额
                amount = singleMin;
            }
            if (overflow(redPacketAmount - amount, redPacketNum -1, minAmount)) {
                //后续红包金额不足最小值
                amount = singleMin;
            }
            if (amount > max || amount < minAmount) {
                //理论上不会出现，最后的保障
                log.error("randomAlgorithm1, amount error, redPacketNum={}, redPacketAmount={}, minAmount={}, amount={}", originalNum, originalAmount, minAmount, amount);
                return Collections.emptyList();
            }
            result.add(amount);
            totalAmount += amount;
            //设置偏移量
            offset = offset + amount - ave;
            //从红包金额、红包个数中扣除本次红包
            redPacketNum--;
            redPacketAmount -= amount;
        }
        if (totalAmount != originalAmount) {
            //理论上不会出现，最后的保障
            log.error("randomAlgorithm1, totalAmount != redPacketAmount, redPacketNum={}, redPacketAmount={}, minAmount={}, totalAmount={}", originalNum, originalAmount, minAmount, totalAmount);
            return Collections.emptyList();
        }
        return result;
    }

    /**
     * 判断当前操作是否会造成
     * 后续红包金额不足最小值
     * @param surplusTotalAmount 剩余红包金额，单位分
     * @param surplusTotalNum 剩余红包个数
     * @param minAmount 单个红包最小金额，单位分
     * @return true-后续金额会不足最小值，false-后续金额充足
     */
    private static boolean overflow(int surplusTotalAmount, int surplusTotalNum, int minAmount) {
        int needTotalAmount = minAmount * surplusTotalNum;
        return surplusTotalAmount < needTotalAmount;
    }
}
