package cn.com.duiba.kjy.base.customweb.autoconfig.graceclose;


import ch.qos.logback.classic.Level;
import ch.qos.logback.classic.LoggerContext;
import cn.com.duiba.boot.event.ContextClosingEvent;
import cn.com.duiba.kjy.base.customweb.sever.CustomNettyServer;
import cn.com.duiba.wolf.threadpool.NamedThreadFactory;
import cn.com.duibaboot.ext.autoconfigure.web.ServerStatusHolder;
import com.alibaba.ttl.threadpool.TtlExecutors;
import com.netflix.discovery.EurekaClient;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.NoSuchBeanDefinitionException;
import org.springframework.boot.SpringApplication;
import org.springframework.boot.SpringApplicationRunListener;
import org.springframework.context.ConfigurableApplicationContext;
import org.springframework.context.event.SimpleApplicationEventMulticaster;
import org.springframework.context.support.AbstractApplicationContext;
import org.springframework.core.env.ConfigurableEnvironment;

import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.IdentityHashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

/**
 * 自动配置优雅停机
 * <br/>
 *
 * 不能优雅停机的原因：各个框架包括spring、dubbo、xmemcached等在初始化时都添加了自己的shutdownhook，而且这些shutdownhook的执行是无序的，所以有可能在dubbo销毁前xmemcached、datasource等已经销毁，导致关闭期间的几秒内大量报错
 * <br/>
 *
 * 解决方法：在spring启动后通过反射移除所有shutdownhook，并添加必要的spring的shutdownhook、添加必要的dubbo清理代码，实现优雅停机
 *
 * Created by wenqi.huang on 2016/12/27.
 */
public class KjjGracefulCloseRunListener implements SpringApplicationRunListener {

    private static final Logger logger = LoggerFactory.getLogger(KjjGracefulCloseRunListener.class);

    private SpringApplication application;

    /**
     * 必须有这个构造函数，否则spring无法初始化该类
     * @param application
     * @param args
     */
    public KjjGracefulCloseRunListener(SpringApplication application, String[] args) {
        this.application = application;
    }

    //@Override //boot新版本没有这个方法，故去掉Override
    public void started() {
        //do nothing
    }

    public void starting() {
    }

    //@Override
    public void environmentPrepared(ConfigurableEnvironment environment) {
        //do nothing
    }

    //@Override
    public void contextPrepared(ConfigurableApplicationContext context) {
        //do nothing
    }

    //@Override
    public void contextLoaded(ConfigurableApplicationContext context) {
        //do nothing
    }

    @Override
    public void started(final ConfigurableApplicationContext context) {
        //自定义的netty服务器才执行后面的操作
        try {
            context.getBean(CustomNettyServer.class);
        }catch (BeansException e){
          return;
        }
        printHomePageUrl(context);

        try {
            //TODO 切换到springboot，而且没用memcached客户端的时候，可能不再需要这个类了，到时试试
            Class clazz = Class.forName("java.lang.ApplicationShutdownHooks");
            Field field = clazz.getDeclaredField("hooks");
            field.setAccessible(true);
            IdentityHashMap<Thread, Thread> hooks = (IdentityHashMap<Thread, Thread>) field.get(null);

            //清空原有hook,但保留日志的hook
            List<Thread> keys2remove = new ArrayList<>();
            if(hooks == null) {
                return;
            }

            for(Map.Entry<Thread, Thread> entry : hooks.entrySet()){
                String hookClassName = entry.getKey().getClass().getName();
                if(hookClassName.equals("java.util.logging.LogManager$Cleaner")
                        || hookClassName.equals("org.unidal.helper.Threads$Manager$1") //cat
                        || hookClassName.startsWith("org.jacoco.agent.rt")//jacoco,去掉shutdownhook这个会导致sonarqube 任务执行失败
                        || hookClassName.startsWith("org.quartz")){
                    continue;
                }
                logger.debug("remove shotdownhook:{}", hookClassName);
                keys2remove.add(entry.getKey());
            }
            for(Thread t : keys2remove) {
                hooks.remove(t);
            }

            addShutDownHook(context);
        } catch (IllegalStateException e) {
            // If the VM is already shutting down,
            // We do not need to register shutdownHook.
        } catch (Exception e){
            logger.error(e.getMessage(), e);
        }
    }

    private void addShutDownHook(ConfigurableApplicationContext context){//NOSONAR
        //添加自己的清理hook
        Runtime.getRuntime().addShutdownHook(new Thread("ShutDownThread") {
            @Override
            public void run() {
                long start = System.currentTimeMillis();

                if(!logger.isInfoEnabled()) {
                    LoggerContext loggerContext = (LoggerContext) LoggerFactory.getILoggerFactory();
                    loggerContext.getLogger(KjjGracefulCloseRunListener.class).setLevel(Level.INFO);
                }

                logger.info("begin to shutdown server...");
                //close Thread in eureka client bean
                context.publishEvent(new ContextClosingEvent(context));

                //先让http接口 [/monitor/check] 返回服务不可用
                logger.info("disable /monitor/check");
                ServerStatusHolder.setInService(false);

                try {
                    Class.forName("com.netflix.discovery.EurekaClient");
                    EurekaClient eurekaClient = context.getBean(EurekaClient.class);
                    if (eurekaClient != null) {
                        logger.info("unregister from eureka-server");
                        //spring的做法在关闭的时候会在线程中调用register，同时调用shutdown方法，由于这两个的顺序无法控制，所以有时候可能先shutdown了，然后又把服务注册上去了。
                        //这里在销毁spring之前取出这个bean先行销毁，以防止此bug
                        eurekaClient.shutdown();
                    }
                }catch(NoSuchBeanDefinitionException | ClassNotFoundException | NoClassDefFoundError e){//EurekaClient 可能部分项目不会引入此类
                    //Ignore
                    logger.warn("unregister from eureka-server failed", e);
                }

                //关闭时处理spring event的顺序不重要，设置线程池，主要是为了拦截一个异常不打印，以解决spring的bug。(详情参考ApplicationEventMulticasterAutoConfiguration类的注释)
                SimpleApplicationEventMulticaster multicaster = context.getBean(AbstractApplicationContext.APPLICATION_EVENT_MULTICASTER_BEAN_NAME, SimpleApplicationEventMulticaster.class);
                if(multicaster != null) {
                    multicaster.setTaskExecutor(TtlExecutors.getTtlExecutorService(new ThreadPoolExecutor(1, 5, 10, TimeUnit.SECONDS, new LinkedBlockingQueue<>(), new NamedThreadFactory("spring-event-multicaster-"))));
                }
                logger.info("sleep 4 seconds before shutdown");
                //虽然GracefulCloseAutoConfiguration中睡了6s 但是线上日志输出的时间显示只有2s。所以这里加4s。
                try {
                    TimeUnit.SECONDS.sleep(4);
                } catch (InterruptedException e) {
                    //Ignore
                    Thread.currentThread().interrupt();
                }
                logger.info("clean up spring begin");
                try {
                    //等待6S的动作会在这个方法内部被调用,见 GracefulCloseAutoConfiguration
                    //内部等待6秒后会销毁dubbo
                    context.close();//再清理spring
                } catch (Throwable e) {
                    logger.error(e.getMessage(), e);
                }

                //gracefully close the threadpool for embedServletContainer

                long period = System.currentTimeMillis() - start;
                logger.info("shutdown server finished successfully，cost：{} ms, bye bye...", period);

                //最后优雅关闭Logback，关闭之后日志无法输出到文件
                LoggerContext loggerContext = (LoggerContext) LoggerFactory.getILoggerFactory();
                loggerContext.stop();
            }
        });
    }

    private void printHomePageUrl(ConfigurableApplicationContext context){
        Class<?> mainApplicationClass = application.getMainApplicationClass();
        //设置mainClass入口的logger级别为INFO, 以让spring打印的启动耗时日志能打印出来
        ch.qos.logback.classic.Logger mainLogger = ((ch.qos.logback.classic.Logger)LoggerFactory.getLogger(mainApplicationClass));
        mainLogger.setLevel(Level.INFO);

        String port = context.getEnvironment().getProperty("server.port");
        String contextPath = context.getEnvironment().getProperty("server.context-path");
        if(StringUtils.isBlank(contextPath)){
            contextPath = context.getEnvironment().getProperty("server.contextPath");
        }
        contextPath = StringUtils.trimToEmpty(contextPath);
        String profile = context.getEnvironment().getProperty("spring.profiles.active");
        mainLogger.info("current profile:{}, home page: http://localhost:{}{}", profile, port, contextPath);
    }

    @Override
    public void running(ConfigurableApplicationContext context) {
        //do nothing
    }

    @Override
    public void failed(ConfigurableApplicationContext context, Throwable exception) {
        //do nothing
    }

}
