package cn.com.duiba.kjy.base.utils;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.zjtlcb.fcloud.utils.MD5Util;
import com.zjtlcb.fcloud.utils.SM2Util;
import com.zjtlcb.fcloud.utils.SM3Util;
import com.zjtlcb.fcloud.utils.SM4Util;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.HttpEntity;
import org.apache.http.HttpResponse;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.ByteArrayEntity;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.util.EntityUtils;

import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.UUID;

/**
 * 泰隆银行工具类-测试环境和生产环境的appId、公钥、私钥、加密密钥都一致
 * @author lizhi
 * @date 2022/11/3 10:15 上午
 */
@Slf4j
public class TlSecurityUtil {

    private TlSecurityUtil() {}

    /**
     * 商户appID
     */
    private static final  String TAI_LONG_APP_ID = "7428f510-4187-4169-b359-a9295fcc6b23";

    /**
     * 泰隆公钥
     */
    private static final String PUBLIC_KEY = "MFkwEwYHKoZIzj0CAQYIKoEcz1UBgi0DQgAE/knoPfu5Dyzevnuxt/W8R3KGZuvzCJz5CTAJkqoWepnpjI2Ar5Q4Ba8y0Qo0WXmUPOd8IQ/kDEPqe43fMtErCg==";

    /**
     * 自有私钥
     */
    private static final String PRIVATE_KEY = "MIGTAgEAMBMGByqGSM49AgEGCCqBHM9VAYItBHkwdwIBAQQgG9IbK5uambvQpn7A47CvmZVbjyD054WVMU/PuB1vje2gCgYIKoEcz1UBgi2hRANCAATe8Z/6BFSMDnIppQUYhbB7B/6Gh7rwXakCQmLs2pMjl587D8SiKdq2HRTRFCIu57uYAU75MfLMfTV6ODUfasVJ";

    /**
     * 加密秘钥
     */
    private static final String APP_SECRET_KEY = "d23b90de-63ae-4325-bf77-7cfa38cac1ef";

    /**
     * 泰隆银行成功状态码
     */
    private static final String SUCCESS_CODE = "000000";

    /**
     * 解密泰隆银行参数
     * @param param 参数
     * @return 解密后的rspData
     */
    public static String parseParamRspData(String param) {
        if (StringUtils.isBlank(param)) {
            return param;
        }
        try {
            JSONObject json = JSON.parseObject(param);
            return parseResult(json, json.getString("appAccessToken"), false, false);
        } catch (Exception e) {
            log.error("TlSecurityUtil, param={}", param, e);
            return param;
        }
    }

    /**
     * 获取泰隆银行业务请求token
     * @param url 接口地址
     * @return token
     */
    public static String getAccessToken(String url) {
        return callTl(url, "approveDev", null, null, true);
    }

    /**
     * 请求泰隆业务接口
     * @param url 接口地址
     * @param serviceId 业务接口ID
     * @param reqData 请求参数
     * @param accessToken 从泰隆获取的token
     * @return 解析后的响应rspData
     */
    public static String callBiz(String url, String serviceId, String reqData, String accessToken) {
        return callTl(url, serviceId, reqData, accessToken, false);
    }

    /**
     * 构建请求报文
     * @param reqData 实际需要传输的数据
     * @param accessToken 与泰隆银行交互的token
     * @return 请求报文
     */
    public static String buildReqParam(String reqData, String accessToken) {
        try {
            JSONObject param = getParam(reqData, accessToken, false);
            return param.toJSONString();
        } catch (Exception e) {
            log.error("TlSecurityUtil, buildReqParam, reqData={}, accessToken={}", reqData, accessToken);
            return "";
        }
    }

    /**
     * 调用泰隆银行接口
     * @param url 请求接口地址
     * @param serviceId 业务接口ID
     * @param reqData 请求参数
     * @param accessToken 从泰隆银行获取的token，获取token接口传null
     * @param isToken 是否是获取token请求
     * @return 响应数据
     */
    private static String callTl(String url, String serviceId, String reqData, String accessToken, boolean isToken) {
        String rspData = null;
        try {
            rspData = doCallTl(url, serviceId, reqData, accessToken, isToken);
            return rspData;
        } catch (Exception e) {
            log.error("TlSecurityUtil, serviceId={}, reqData={}, accessToken={}, isToken={}", serviceId, reqData, accessToken, isToken);
            return null;
        } finally {
            log.info("TlSecurityUtil, response data, serviceId={}, reqData={}, accessToken={}, isToken={}, rspData={}", serviceId, reqData, accessToken, isToken, rspData);
        }
    }

    /**
     * 调用泰隆银行接口
     * @param url 请求接口地址
     * @param serviceId 业务接口ID
     * @param reqData 请求参数
     * @param accessToken 从泰隆银行获取的token，获取token接口传null
     * @param isToken 是否是获取token请求
     * @return 响应数据
     * @throws Exception 请求过程中发生的异常
     */
    private static String doCallTl(String url, String serviceId, String reqData, String accessToken, boolean isToken) throws Exception {
        // 获取请求参数
        JSONObject param = getParam(reqData, accessToken, isToken);
        log.info("TlSecurityUtil, getParam, serviceId={}, reqData={}, accessToken={}, isToken={}, param={}", serviceId, reqData, accessToken, isToken, param);
        // 调用泰隆银行
        String rspMsg = doPost(param, url + serviceId);
        log.info("TlSecurityUtil, response, url={}, reqData={}, accessToken={}, isToken={}, param={}, rspMsg={}", url + serviceId, reqData, accessToken, isToken, param, rspMsg);
        // 解析银行返回数据
        return parseResult(rspMsg, accessToken, isToken);
    }

    /**
     * 封装加密后的请求参数
     * @param reqData 业务请求参数，获取token接口传null
     * @param accessToken 从泰隆银行获取的token，获取token接口传null
     * @param isToken  是否是获取token请求
     * @return 封装加密后的完整请求参数
     * @throws Exception 加密过程中可能会抛出异常
     */
    private static JSONObject getParam(String reqData, String accessToken, boolean isToken) throws Exception {
        String seqNo = (new SimpleDateFormat("yyyyMMddHHmmsss")).format(new Date());
        if (!isToken) {
            // 获取业务请求参数
            return getSignParam(reqData, accessToken, false, seqNo);
        }
        // 获取token请求参数
        String random = MD5Util.md5_(seqNo);
        JSONObject param = getSignParam(random, accessToken, true, seqNo);
        param.put("random", random);
        return param;
    }

    /**
     * 获取加密好的请求参数
     * @param data 业务请求参数
     * @param accessToken 从泰隆银行获取的token，获取token接口传null
     * @param isToken 是否是获取token
     * @param seqNo 流水号
     * @return 请求参数
     * @throws Exception 加密方法可能会抛出异常
     */
    private static JSONObject getSignParam(String data, String accessToken, boolean isToken, String seqNo) throws Exception {
        JSONObject param = new JSONObject();
        param.put("appID", TAI_LONG_APP_ID);
        param.put("seqNO", seqNo);
        String randomKey = MD5Util.md5_(UUID.randomUUID().toString());
        param.put("sm2EncryptData", SM2Util.encryptByPublicKey(randomKey, PUBLIC_KEY));
        param.put("sm2Sign", SM2Util.signByPrivateKey(randomKey, PRIVATE_KEY, TAI_LONG_APP_ID));
        param.put("sign", SM3Util.sign(data + seqNo + APP_SECRET_KEY + randomKey));
        if (isToken) {
            return param;
        }
        // 业务参数，需要对请求参数加密
        param.put("signMethod", "SM3");
        param.put("encryptMethod", "SM4");
        param.put("appAccessToken", accessToken);
        param.put("reqData", SM4Util.encrypt(data, seqNo + accessToken + APP_SECRET_KEY + randomKey));
        return param;
    }

    /**
     * 调用泰隆银行
     * @param param 请求参数
     * @param url 请求接口
     * @return 泰隆银行返回结果
     * @throws IOException http请求可能会抛出IO异常
     */
    private static String doPost(JSONObject param, String url) throws IOException {
        String msg = param.toJSONString();
        HttpClient httpClient = HttpClientBuilder.create().build();
        HttpPost httpPost = new HttpPost(url);
        ByteArrayEntity bae = new ByteArrayEntity(msg.getBytes());
        httpPost.setEntity(bae);
        httpPost.addHeader("Content-Type", "application/x-www-form-urlencoded; charset=UTF-8");
        HttpResponse response = httpClient.execute(httpPost);
        HttpEntity entity = response.getEntity();
        if (entity != null) {
            byte[] bytes = EntityUtils.toByteArray(entity);
            return new String(bytes, StandardCharsets.UTF_8);
        } else {
            return null;
        }
    }

    /**
     * 解析泰隆银行响应结果
     * @param rspMsg 泰隆银行响应结果
     * @param accessToken 从泰隆银行获取的token，获取token接口传null
     * @param isToken 是否是获取token
     * @return 响应数据
     * @throws Exception 验签时可能会抛出异常
     */
    private static String parseResult(String rspMsg, String accessToken, boolean isToken) throws Exception {
        JSONObject rspJsonObj = JSON.parseObject(rspMsg);
        String errorCode = rspJsonObj.getString("errorCode");
        if (!SUCCESS_CODE.equals(errorCode) && isToken) {
            // 业务上的接口errorCode不是000000的时候，也代表请求成功
            log.info("TlSecurityUtil, code error, rspMsg={}", rspMsg);
            return null;
        }
        return parseResult(rspJsonObj, accessToken, isToken, true);
    }

    private static String parseResult(JSONObject json, String accessToken, boolean isToken, boolean isRsp) throws Exception {
        // 使用私钥 从 sm2EncryptData 获取随机密码
        String randomKey = SM2Util.decryptByPrivateKey(json.getString("sm2EncryptData"), PRIVATE_KEY);
        // 使用公钥验证签名
        boolean sm2Sign = SM2Util.verifyByPublicKey(json.getString("sm2Sign"), PUBLIC_KEY, TAI_LONG_APP_ID, randomKey);
        if (!sm2Sign) {
            log.info("TlSecurityUtil, sm2 sign error, json={}, randomKey={}", json.toJSONString(), randomKey);
            return null;
        }
        String seqNo = json.getString("seqNO");
        if (isToken) {
            // 解析响应的token
            return parseToken(json, randomKey, seqNo);
        }
        // 解析业务请求的响应
        return parseBiz(json, accessToken, randomKey, seqNo, isRsp);
    }

    private static String parseToken(JSONObject rspJsonObj, String randomKey, String seqNo) {
        boolean verify = SM3Util.verify(rspJsonObj.getString("random") + seqNo + randomKey + APP_SECRET_KEY, rspJsonObj.getString("sign"));
        if (!verify) {
            log.info("TlSecurityUtil, get token, sm3 sign error, rspMsg={}, randomKey={}", rspJsonObj.toJSONString(), randomKey);
            return null;
        }
        // 获取token的接口，token是放在sm2EncryptData里的
        return randomKey;
    }

    private static String parseBiz(JSONObject json, String accessToken, String randomKey, String seqNo, boolean isRsp) {
        String data = isRsp ? json.getString("rspData") : json.getString("reqData");
        // 使用 SM4 对响应数据 rspData 进行解密
        String rspData = SM4Util.decrypt(data, seqNo + accessToken + APP_SECRET_KEY + randomKey);
        // 验证签名
        boolean verify = SM3Util.verify(rspData + seqNo + APP_SECRET_KEY + randomKey, json.getString("sign"));
        if (!verify) {
            log.info("TlSecurityUtil, sm3 sign error, json={}, randomKey={}, accessToken={}", json.toJSONString(), randomKey, accessToken);
            return null;
        }
        return rspData;
    }
}
