package cn.com.duiba.kjy.base.api.utils;

import cn.com.duiba.boot.utils.SpringEnvironmentUtils;
import cn.com.duiba.wolf.utils.BlowfishUtils;
import com.alibaba.fastjson.JSON;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;

import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 登录相关工具
 * @author lizhi
 * @date 2021/12/2 4:23 下午
 */
@Slf4j
public class LoginUtil {

    private LoginUtil() {}

    /**
     * 将对象转换成cookie值
     * @param dto 对象
     * @param keyEncrypt 密钥
     * @param <T> 对象类型
     * @return cookie值
     */
    public static <T extends Serializable> String createCookieValue(T dto, String keyEncrypt) {
        if (dto == null || StringUtils.isBlank(keyEncrypt)) {
            return null;
        }
        try {
            String value = JSON.toJSONString(dto);
            return BlowfishUtils.encryptBlowfish(value, keyEncrypt);
        } catch (Exception e) {
            log.error("LoginUtil, get cookie value error, dto={}, keyEncrypt={}", dto, keyEncrypt, e);
            return null;
        }
    }

    /**
     * 根据cookie名称，获取cookie的值
     * @param request http请求
     * @param cookieName cookie名称
     * @return cookie值
     */
    public static String getCookieValue(HttpServletRequest request, String cookieName) {
        Cookie cookie = getOneCookieByName(request, cookieName);
        if (cookie == null) {
            return null;
        }
        return cookie.getValue();
    }

    /**
     * 解析cookie
     * @param cookieValue cookie值
     * @param keyEncrypt 密钥
     * @param clazz cookie原始对象class
     * @param <T> cookie原始对象类型
     * @return cookie原始对象
     */
    public static <T extends Serializable> T decodeCookie(String cookieValue, String keyEncrypt, Class<T> clazz){
        if (StringUtils.isBlank(cookieValue) || StringUtils.isBlank(keyEncrypt) || clazz == null){
            return null;
        }
        try {
            String content = BlowfishUtils.decryptBlowfish(cookieValue, keyEncrypt);
            return JSON.parseObject(content, clazz);
        } catch (Exception e) {
            log.error("LoginUtil, decode cookie error, cookieValue={}, keyEncrypt={}, className={}", cookieValue, keyEncrypt, clazz.getName(), e);
            return null;
        }
    }

    /**
     * 添加或更新cookie
     * @param response http响应
     * @param domains 需要添加的域名集合
     * @param cookieName cookie名称
     * @param cookieValue cookie值
     * @param expiredDays cookie有效期天数
     * @return 是否成功
     */
    public static boolean addOrUpdateCookie(HttpServletRequest request, HttpServletResponse response, List<String> domains, String cookieName, String cookieValue, int expiredDays) {
        if (request == null || response == null || CollectionUtils.isEmpty(domains) || StringUtils.isBlank(cookieName) || StringUtils.isBlank(cookieValue) || expiredDays < 1) {
            log.error("LoginUtil, addCookie param error, cookieName={}, cookieValue={}, expiredDays={}, domains.size={}", cookieName, cookieValue, expiredDays, domains == null ? 0 : domains.size());
            return false;
        }
        try {
            doAddOrUpdateCookie(request, response, domains, cookieName, cookieValue, expiredDays);
            return true;
        } catch (Exception e) {
            log.error("LoginUtil, addCookie error, cookieName={}, cookieValue={}, expiredDays={}, domains.size={}, e:", cookieName, cookieValue, expiredDays, domains.size(), e);
            domains.forEach(domain -> log.error("LoginUtil, doLoginCookie error, domain={}", domain));
            return false;
        }
    }

    /**
     * 清除cookie
     * @param request http请求
     * @param response http响应
     * @param domains 需要清除的域名集合
     * @param cookieName cookie名称
     * @return 是否成功
     */
    public static boolean cleanCookie(HttpServletRequest request, HttpServletResponse response, List<String> domains, String cookieName) {
        if (request == null || response == null || CollectionUtils.isEmpty(domains) || StringUtils.isBlank(cookieName)) {
            log.error("LoginUtil, cleanCookie param error, cookieName={}, domains.size={}", cookieName, domains == null ? 0 : domains.size());
            return false;
        }
        try {
            doUpdateCookie(request, response, domains, cookieName, null, 0);
            return true;
        } catch (Exception e) {
            log.error("LoginUtil, cleanCookie error, cookieName={}, domains.size={}, e:", cookieName, domains.size(), e);
            domains.forEach(domain -> log.error("LoginUtil, cleanCookie error, domain={}", domain));
            return false;
        }
    }


    private static void doAddOrUpdateCookie(HttpServletRequest request, HttpServletResponse response, List<String> domains, String cookieName, String cookieValue, int expiredDays) {
        int maxAge = expiredDays * 24 * 60 * 60;
        boolean updateResult = doUpdateCookie(request, response, domains, cookieName, cookieValue, maxAge);
        if (updateResult) {
            // 已经变更过值了，不需要新增了
            return;
        }
        // 之前没有该cookie，添加cookie
        Cookie userCookie = new Cookie(cookieName, cookieValue);
        fillCookie(userCookie, maxAge);
        // 循环添加域名
        domains.forEach(domain-> addCookie(response, userCookie, domain));
        if (SpringEnvironmentUtils.isProdEnv()) {
            return;
        }
        // 非生产环境，加入到header
        response.addHeader(cookieName, cookieValue);
    }

    private static boolean doUpdateCookie(HttpServletRequest request, HttpServletResponse response, List<String> domains, String cookieName, String cookieValue, int maxAge) {
        List<Cookie> cookieList = getCookieByName(request, cookieName);
        if (CollectionUtils.isEmpty(cookieList)) {
            return false;
        }
        cookieList.forEach(cookie -> doUpdateCookie(response, cookie, domains, cookieValue, maxAge));
        return true;
    }

    private static Cookie getOneCookieByName(HttpServletRequest request, String cookieName) {
        List<Cookie> cookieList = getCookieByName(request, cookieName);
        if (CollectionUtils.isEmpty(cookieList)) {
            return null;
        }
        // 返回最后一个
        return cookieList.get(cookieList.size() - 1);
    }

    private static List<Cookie> getCookieByName(HttpServletRequest request, String cookieName) {
        if (request == null) {
            return Collections.emptyList();
        }
        Cookie[] cookies = request.getCookies();
        if (cookies == null) {
            return Collections.emptyList();
        }
        return Arrays.stream(cookies).filter(cookie -> StringUtils.equals(cookie.getName(), cookieName)).collect(Collectors.toList());
    }

    private static void doUpdateCookie(HttpServletResponse response, Cookie cookie, List<String> domains, String cookieValue, int maxAge) {
        fillCookie(cookie, maxAge, cookieValue);
        domains.forEach(domain-> addCookie(response, cookie, domain));
    }

    private static void addCookie(HttpServletResponse response, Cookie cookie, String domain) {
        cookie.setDomain(domain);
        response.addCookie(cookie);
    }

    private static void fillCookie(Cookie cookie, int maxAge) {
        cookie.setHttpOnly(true);
        cookie.setPath("/");
        cookie.setMaxAge(maxAge);
    }

    private static void fillCookie(Cookie cookie, int maxAge, String cookieValue) {
        fillCookie(cookie, maxAge);
        cookie.setValue(cookieValue);
    }
}
