package cn.com.duiba.kjy.base.utils;

import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.BiFunction;
import java.util.function.Function;

@Slf4j
public class BaseListUtils {


    /**
     * 将两个集合合并为一个map，按照list的顺序一一对应。
     * 适合 redis 和 mybatis的批量操作后，将入参和返回值放进map一同返回
     * @param keys 返回map的key集合
     * @param values 返回map的value原始数据集合
     * @param valueHandle 可根据实际需求对map的value进行数据转换
     * @return 合并后的map
     */
    public static <T,K,V> Map<K,V> assembleDoubleList2Map(List<K> keys, List<T> values, BiFunction<K,T,V> valueHandle){
        return assembleDoubleList2Map(new HashMap<>(),keys,values,Function.identity(),valueHandle);
    }

    /**
     * 将两个集合合并为一个map，按照list的顺序一一对应。
     * 适合 redis 和 mybatis的批量操作后，将入参和返回值放进map一同返回
     * @param keys 返回map的key集合
     * @param values 返回map的value集合
     * @return 合并后的map
     */
    public static <K,V> Map<K,V> assembleDoubleList2Map(List<K> keys, List<V> values){
        return assembleDoubleList2Map(new HashMap<>(),keys,values,Function.identity(),(k,v)->v);
    }

    /**
     * 合并两个list为一个key value的map
     * @param map 初始KV集合，可放进一些已经有结果的数据
     * @param keys 返回map的key原数据集合
     * @param values 返回map的value原数据集合
     * @param keyHandle key的转换器，可更具需求转换key值
     * @param valueHandle value转换器，可更具需求转换value值
     * @return 合并后的map
     * @param <T> 返回值map中value的原始数据类型
     * @param <P> 返回值map中key的原始数据类型
     * @param <K> 返回值map中key的最终类型
     * @param <V> 返回值map中value的最终类型
     */
    public static <T,P,K,V> Map<K,V> assembleDoubleList2Map(Map<K,V> map,List<P> keys, List<T> values, Function<P,K> keyHandle, BiFunction<K,T,V> valueHandle){
        if (CollectionUtils.isEmpty(keys) || CollectionUtils.isEmpty(values) || keys.size()!=values.size()){
            log.error("the list size of keys and values is  unequals!");
            return null;
        }
        for (int i = 0; i< keys.size(); i++) {
            K key = keyHandle.apply(keys.get(i));
            V value = valueHandle.apply(key,values.get(i));
            map.put(key,value);
        }
        return map;
    }
}
