package cn.com.duiba.kjy.base.api.utils;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.math.NumberUtils;
import org.jetbrains.annotations.Nullable;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Supplier;

/**
 * @author dugq
 * @date 2021/6/9 4:57 下午
 * 大json，读取属性工具
 * 支持数组，对象，字符串（基本类型可以通过string获取） 在获取到具体的string以后，你可以使用fastJson 进行对象转换等操作
 * exp: JSONPath jsonPath = new JSONPath("
 * {
 *     "A":"a",
 *     "B":[
 *         {
 *             "C":"c1"
 *         },
 *         {
 *             "C":"c2",
 *             "D":"d"
 *         }
 *     ],
 *     "E":{
 *         "F":123,
 *         "G":"g"
 *     },
 *     "H":{
 *         "I":"i",
 *         "J":""
 *     }
 * }
 * ")
 *
 *         jsonPath.getString("A",null);                  a
 *         jsonPath.getString("B[]",null);                [{"C":"c1"},{"C":"c2","D":"d"}]
 *         jsonPath.getString("B[1]",null);               {"C":"c2","D":"d"}
 *         jsonPath.getString("B[1].C",null);             c2
 *         jsonPath.getString("B[].C",null);              ["c1","c2"]
 *         jsonPath.getString("B[].{C}",null);            [{"C":"c1"},{"C":"c2"}]
 *         jsonPath.getString("{A,B[].C,H.I}",null);      {"A":"a","I":"i"}
 *
 */
@Slf4j
public class JSONPath {

    private JSONObject jsonObject;

    private  JSONArray jsonArray;

    private Object otherObj;

    //原生json的类型
    private final JsonType jsonType;

    private Object obj;

    private static final Supplier DEFAULT_GET = ()-> null;

    //层级间隔符
    private static final char LEVEL_SEP = '.';
    //属性间隔符
    private static final char prop_sep = ',';

    //数组开始标示
    private static final char array_start = '[';
    //数组结束标示
    private static final char array_end = ']';
    //对象开始标志
    private static final char obj_start = '{';
    //对象结束标记
    private static final char obj_end = '}';

    private static final List<Character> important_chars = Arrays.asList(array_start,array_end,obj_start,obj_end,LEVEL_SEP,prop_sep);

    //递归计数器
    private Integer recursionCount = 0;
    //避免死循环递归.也许当您的json足够大时，它可能会不适用，您可以根据实际情况进行调整。但是，务必存在，否则可能将会带来意想不到的灾难
    private Integer maxRecursionCount = 500;

    enum JsonType{
        OBJ, //jsonObject
        ARRAY, //jsonArray
        OTHER //其他类型，包括：基本类型和字符串。此类就是个字符串，有什么好解析的
    }

    public JSONPath(String jsonStr){
        final Object parse = JSON.parse(jsonStr);
        if (parse instanceof JSONObject){
            jsonObject = (JSONObject)parse;
            jsonType = JsonType.OBJ;
        }else if(parse instanceof JSONArray){
            jsonArray = (JSONArray)parse;
            jsonType = JsonType.ARRAY;
        }else{
            otherObj = parse;
            jsonType = JsonType.OTHER;
        }
    }

    public boolean isJSON(){
        return jsonType == JsonType.OBJ || jsonType == JsonType.ARRAY;
    }

    /**
     * 从json中读取指定属性组成的字符串
     *
     * @param fullPath 完整路径。
     *                 1、你可以使用.进行路径分割，例如："A.B",表示获取A属性下的B属性的值<br/>
     *                 2、您可以使用[]来获取数组中的值，例如："B[1].C"表示获取数组中下标为1的对象中的C属性的值，
     *                 当然你也可以不填下表，那么本方法将返回整个数组中满足您要求的值。例如"B[].C"，
     *                 本方法将返回一个jsonArray，包含B数组下所有对象的C属性的值<br/>
     *                 3、您也可以通过{}来获取一个对象。例如：{A,B[].C,H.I} ，多个属性用,个隔开,
     *                 每个属性的路径可以有多级，我们将以最后一级的Key作为返回对象的key。
     *                 您可以通过此方法提取出一个大json中您需要的key-value，然后再此使用fastJson将字符串转化为对象
     *                 例如：<br/>
     *                     JSONPath p = new JSONPath("假设这是一个很大的json");
     *                     String json = p.getString("{A.B.C.D, E.F.G.H, I.K.M.L}");
     *                     YourClass result = JSONObject.parseObject(json,YourClass.class);
     *                 这将极大的改善您对json结果的获取方式。
     * @param elseGet 当您的路径有误，或者传入的json和您预想的json不一样时，elseGet将返回一个您需要的默认值，当然，您可以传null，
     *                本方法业务智能的填充一个默认的 Supplier，它将返回null关键字。如果它不是您想要的，你可以自行定制。
     * @return 本方法根据您传入的路径智能处理，您可以需要一个array，string or an object。
     *         我们最终都将其序列化为字符串。当然也可能是elseGet返回的值。
     */
    public String getString(String fullPath , Supplier<String> elseGet) {
        recursionCount = 0;
        if (Objects.isNull(elseGet)){
            elseGet = DEFAULT_GET;
        }
        if(StringUtils.isBlank(fullPath)){
            return elseGet.get();
        }
        Object result;
        try {
            result = getObject(fullPath);
        }catch (Exception e){
            result = null;
        }
        return Objects.isNull(result)?elseGet.get():JSON.toJSONString(result);
    }

    /**
     * 这是一个基本等价于getString的方法获取Object时的样子，
     * 也许您不想把它作为字符串返回，只是想对jsonObject进行一次利用，您可以选择使用此方法。
     */
    public JSONObject getObject(String fullPath , Supplier<JSONObject> elseGet) {
        recursionCount = 0;
        if (Objects.isNull(elseGet)){
            elseGet = DEFAULT_GET;
        }
        if(StringUtils.isBlank(fullPath)){
            return elseGet.get();
        }
        Object result;
        try {
            result = getObject(fullPath);
        }catch (Exception e){
            result = null;
        }
        return result instanceof JSONObject ? (JSONObject) result : elseGet.get();
    }


    @Nullable
    private Object getObject(String fullPath) {
        final List<Path> paths = explainPattern(fullPath);
        switch (jsonType){
            case OBJ:return doExplainByLevel(paths, jsonObject);
            case ARRAY:return doExplainByLevel(paths, jsonArray);
            default:return null;
        }
    }

    @Nullable
    private Object doExplainByLevel(List<Path> sortedPath, Object result) {
        if (recursionCount++> maxRecursionCount){
            throw new RuntimeException("超过最大深度");
        }
        for (int i = 0; i < sortedPath.size(); i++) {
            Path currentPath = sortedPath.get(i);
            if (Objects.isNull(result)) {
                break;
            } else if (result instanceof JSONObject) {
                result = getObjectFromObject((JSONObject)result, currentPath);
            } else if (result instanceof JSONArray) {
                if (currentPath.type==PathType.ARRAY && StringUtils.isBlank(currentPath.value)){
                    //取完整List的时候，就不能再顺序执行了，需要再嵌套一层循环，取合集。外层循环自动终止
                    result = getArrayFromArray(sortedPath.subList(i+1,sortedPath.size()),(JSONArray)result);
                    break;
                }else if (currentPath.type==PathType.ARRAY){
                    result = getObjectFromArray(currentPath.value,(JSONArray)result);
                }else{ //不能从数组中取其他类型哦
                    result = null;
                }
            } else {
                // 其他类型，无法get，统一返回 null
                return null;
            }

        }
        return result;
    }

    private Object getObjectFromObject(JSONObject result, Path currentPath) {
        if(currentPath.type == PathType.OBJECT){
            JSONObject jsonObject = new JSONObject();
            for (List<Path> props : currentPath.subPath) {
                jsonObject.put(props.get(props.size()-1).value,doExplainByLevel(props,result));
            }
            return jsonObject;
        }else if (currentPath.type==PathType.String){
            return result.get(currentPath.value);
        }else{
            return null;
        }
    }

    private Object getArrayFromArray(List<Path> currentPath, JSONArray result) {
        JSONArray arrayResult = new JSONArray();
        for (Object subResult : result) {
            arrayResult.add(doExplainByLevel(currentPath, subResult));
        }
        return arrayResult;
    }

    private Object getObjectFromArray(String currentPath,JSONArray result){
        if (NumberUtils.isCreatable(currentPath)){
            return result.get(Integer.parseInt(currentPath));
        }else{
            return null;
        }
    }

    /**
     * 解析json path字符串
     * 为了避免字符串的过多遍历，以及split 方法的混乱使用，这里使用游标算法，对输入字符串进行一次遍历，获取去全路径
     * 在json对象较深的情况下拥有较深的意义
     */
    private List<Path> explainPattern(String fullPath) {
        AtomicInteger index = new AtomicInteger();

        try {
            final char[] chars = fullPath.toCharArray();
            List<Path> pathList = new ArrayList<>();
            for (;index.get()<chars.length;) {
                if (chars[index.get()]==LEVEL_SEP){
                    index.incrementAndGet();
                }
                pathList.add(builderPath(chars,index));
            }
            return pathList;
        }catch (Exception e){
            log.error("",e);
            throw new RuntimeException("illegal char at "+ index.get()+ "value = "+fullPath.charAt(index.get()) +" in json string = "+fullPath);
        }
    }

    private Path builderPath(char[] sourceChars,AtomicInteger index){
        char currentChar= sourceChars[index.get()];
        if (currentChar ==obj_start){
            final Path path = buildObjectPath(sourceChars, index);
            return path;
        }else if (currentChar==array_start){
            final Path path = buildArrayPath(sourceChars, index);
            return path;
        }else{
            final Path path = buildStringPath(sourceChars, index);
            return path;
        }
    }

    private Path buildObjectPath(char[] sourceChars, AtomicInteger index) {
        char currentChar = sourceChars[index.get()]; //此步骤也许没啥用，但是你应该知道，它标示了当前游标位置上的字符 而且必定是字符 '{'
        List<List<Path>> subPathList = new ArrayList<>();
        List<Path> subPath = new ArrayList<>();
        while (index.incrementAndGet()<sourceChars.length){ //游标始终移动
            subPath.add(builderPath(sourceChars, index));
            if ((currentChar = sourceChars[index.get()])==prop_sep || currentChar == obj_end){
                subPathList.add(subPath);
                subPath = new ArrayList<>(); //为下一个属性链表做准备
            }
            if (currentChar== obj_end){ //结束当前对象
                index.incrementAndGet(); // 跳过对象结束符 '}'
                break;
            }
        }
        Path currentPath = new Path(null,PathType.OBJECT);
        currentPath.subPath = subPathList;
        return currentPath;
    }

    private Path buildArrayPath(char[] sourceChars, AtomicInteger index) {
        int start = index.get()+1; //+1 是为了跳过 '['
        char currentChar = sourceChars[index.get()]; //此步骤也许没啥用，但是你应该知道，它标示了当前游标位置上的字符
        while (index.incrementAndGet()<sourceChars.length && (currentChar = sourceChars[index.get()]) != array_end){
            //do nothing   just make index increment
        }
        String arrayIndex = new String(ArrayUtils.subarray(sourceChars, start, index.get()));
        if (currentChar==array_end){
            index.incrementAndGet(); //当前位置是 ']'，它应该属于数组路径中需要处理掉的字符
        }
        return new Path(arrayIndex,PathType.ARRAY);
    }

    private Path buildStringPath(char[] sourceChars, AtomicInteger index) {
        int start = index.get();
        while (index.incrementAndGet()<sourceChars.length && !important_chars.contains(sourceChars[index.get()])){
             //do nothing   just make index increment
        }
        final String Key = new String(ArrayUtils.subarray(sourceChars, start, index.get()));
        return new Path(Key,PathType.String);
    }

    class Path{
        String value; //it will be a number or blank string when type equals ARRAY. Method return an array when value is blank string.
        PathType type;
        List<List<Path>> subPath;//just enable on type equals OBJECT

        public Path(String value, PathType type) {
            this.value = value;
            this.type = type;
        }
    }

    enum PathType{
        String,
        ARRAY,
        OBJECT
    }

    public void setMaxRecursionCount(int maxRecursionCount){
        this.maxRecursionCount = maxRecursionCount;
    }
}
