package cn.com.duiba.kjy.base.customweb.web.factory;

import cn.com.duiba.kjy.base.common.bean.CrossDomain;
import cn.com.duiba.kjy.base.customweb.autoconfig.CorsDomainConfig;
import cn.com.duiba.kjy.base.customweb.autoconfig.MappingCrosDomainConfig;
import cn.com.duiba.kjy.base.customweb.web.bean.ParameterBean;
import cn.com.duiba.kjy.base.customweb.web.handler.mapping.controller.ControllerMappingHandler;
import cn.com.duiba.kjy.base.customweb.web.handler.mapping.controller.ControllerMappingHandlerBuilder;
import cn.com.duiba.kjy.base.customweb.web.handler.mapping.controller.ControllerMappingRegister;
import cn.com.duiba.kjy.base.customweb.web.handler.mapping.controller.RequestMappingInfo;
import cn.com.duiba.kjy.base.customweb.web.handler.response.ResponseHandler;
import cn.com.duiba.kjy.base.customweb.web.processor.parameter.ParameterPostProcessor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.ArrayUtils;
import org.apache.commons.lang.BooleanUtils;
import org.springframework.beans.factory.BeanInitializationException;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.context.support.ApplicationObjectSupport;
import org.springframework.core.MethodIntrospector;
import org.springframework.core.MethodParameter;
import org.springframework.core.annotation.AnnotatedElementUtils;
import org.springframework.stereotype.Controller;
import org.springframework.util.ClassUtils;
import org.springframework.util.ReflectionUtils;
import org.springframework.web.bind.annotation.RequestMapping;

import javax.annotation.Resource;
import java.lang.reflect.Method;
import java.lang.reflect.Parameter;
import java.util.*;

/**
 * @author dugq
 * @date 2021/3/23 7:44 下午
 */
@Slf4j
public class HandlerMappingBeanFactory extends ApplicationObjectSupport implements InitializingBean {

    @Resource
    private ControllerMappingRegister controllerMappingRegister;
    @Resource
    private CorsDomainConfig corsDomainConfig;

    private final List<ParameterPostProcessor> parameterPostProcessors;

    private final List<ResponseHandler> responseHandlers;

    public HandlerMappingBeanFactory(List<ParameterPostProcessor> parameterPostProcessors, List<ResponseHandler> responseHandlers) {
        this.parameterPostProcessors = parameterPostProcessors;
        this.responseHandlers = responseHandlers;
    }


    public void builderHandlerMappingAndRegister(Class<?> handlerClass, Method method, String handler, RequestMappingInfo requestMappingInfo) throws Exception{
        final ControllerMappingHandler controllerMappingHandler = buildOneHandlerMapping(handlerClass, method,handler,requestMappingInfo);
        controllerMappingRegister.registerHandler(controllerMappingHandler);
    }

    private ControllerMappingHandler buildOneHandlerMapping(Class<?> handlerClass, Method handlerMappingMethod, String handler, RequestMappingInfo requestMappingInfo) throws Exception{
        ControllerMappingHandlerBuilder controllerMappingHandlerBuilder = ControllerMappingHandlerBuilder.create(handlerClass, handlerMappingMethod);
        controllerMappingHandlerBuilder.setRequestMappingInfo(requestMappingInfo);
        final Class<?> returnType = handlerMappingMethod.getReturnType();
        controllerMappingHandlerBuilder.setReturnType(returnType);
        controllerMappingHandlerBuilder.setResponseHandler(getResponseHandler(handlerMappingMethod,returnType));
        controllerMappingHandlerBuilder.setHandler(handler);
        fillCrossDomainConfig(handlerClass,handlerMappingMethod,controllerMappingHandlerBuilder);
        fillParameterList(handlerClass, handlerMappingMethod, controllerMappingHandlerBuilder);
        return controllerMappingHandlerBuilder.build();
    }

    private void fillCrossDomainConfig(Class<?> handlerClass, Method handlerMappingMethod, ControllerMappingHandlerBuilder controllerMappingHandlerBuilder) throws Exception{
        CrossDomain annotation = handlerMappingMethod.getAnnotation(CrossDomain.class);
        if (Objects.isNull(annotation)){
            annotation = handlerClass.getAnnotation(CrossDomain.class);
        }
        if (annotation == null && CollectionUtils.isEmpty(corsDomainConfig.getAllowDomains())) {
            // 没有跨域注解，也没有配置通用的跨域域名
            return;
        }
        controllerMappingHandlerBuilder.setConfig(buildCorsDomainConfig(annotation,controllerMappingHandlerBuilder));
    }

    private MappingCrosDomainConfig buildCorsDomainConfig(CrossDomain crossDomain, ControllerMappingHandlerBuilder controllerMappingHandlerBuilder) throws Exception{
        MappingCrosDomainConfig config = new MappingCrosDomainConfig();
        String[] origins = crossDomain == null ? null : crossDomain.origins();
        if (ArrayUtils.isEmpty(origins)){
            if (CollectionUtils.isNotEmpty(corsDomainConfig.getAllowDomains())){
                config.setAllowedOrigins(corsDomainConfig.getAllowDomains());
            }else{
                throw new BeanInitializationException("@CrossDomain 注解 origins不能为空。跨域必须指定允许的域名。beanName = "+controllerMappingHandlerBuilder.getHandler());
            }
        }else{
            config.setAllowedOrigins(Arrays.asList(origins));
        }
        config.setExposedHeaders(crossDomain == null ? null : Arrays.asList(crossDomain.exposedHeaders()));
        config.setAllowCredentials(crossDomain == null || BooleanUtils.toBoolean(crossDomain.allowCredentials()));
        config.setMaxAge(crossDomain == null ? 3600 : crossDomain.maxAge());
        List<String> allowedHeaders = crossDomain == null ? null : Arrays.asList(crossDomain.allowedHeaders());
        if (CollectionUtils.isEmpty(allowedHeaders)){
            if (corsDomainConfig.isAllowAllHeaderDefault()){
                allowedHeaders = new ArrayList<>();
                allowedHeaders.add("*");
                config.setAllowedHeaders(allowedHeaders);
            }else{
                throw new BeanInitializationException("@CrossDomain allowedHeaders 不能为空。beanName="+controllerMappingHandlerBuilder.getHandler());
            }
        }else{
            config.setAllowedHeaders(allowedHeaders);
        }
        return config;
    }


    private void fillParameterList(Class<?> handlerClass, Method handlerMappingMethod, ControllerMappingHandlerBuilder controllerMappingHandlerBuilder) {
        final Parameter[] parameters = handlerMappingMethod.getParameters();
        List<ParameterBean> parameterBeans = new ArrayList<>();
        int index = 0;
        for (Parameter parameter : parameters) {
            ParameterBean parameterBean = new ParameterBean();
            final Class<?> type = parameter.getType();
            parameterBean.setType(type);
            parameterBean.setName(parameter.getName());
            parameterBean.setPrimitive(type.isPrimitive());
            parameterBean.setMethodParameter(new MethodParameter(handlerMappingMethod,index++));
            processorParameter(parameterBean, handlerClass, handlerMappingMethod);
            parameterBeans.add(parameterBean);
        }
        controllerMappingHandlerBuilder.setParamList(parameterBeans);
    }

    private void processorParameter(ParameterBean parameterBean, Class<?> handlerBeanClass, Method method) {
        for (ParameterPostProcessor parameterPostProcessor : parameterPostProcessors) {
            parameterPostProcessor.postProcessorParameter(parameterBean,handlerBeanClass,method);
        }
    }


    private ResponseHandler getResponseHandler(Method method, Class<?> returnType){
        for (ResponseHandler responseHandler : responseHandlers) {
            if (responseHandler.canWrite(method,returnType)){
                return responseHandler;
            }
        }
        throw new UnsupportedOperationException("can not write this type "+returnType.getName()+" in method "+method.getName());
    }

    @Override
    public void afterPropertiesSet() throws Exception {
        init();
    }

    private void init() throws Exception{
        final String[] list = obtainApplicationContext().getBeanNamesForType(Object.class);
        for (String beanName : list){

            final Class<?> beanClass = obtainApplicationContext().getType(beanName);
            if (Objects.isNull(beanClass)){
                continue;
            }
            if (!AnnotatedElementUtils.hasAnnotation(beanClass, Controller.class)){
                continue;
            }
            Class<?> userType = ClassUtils.getUserClass(beanClass);
            Set<Method> methods = MethodIntrospector.selectMethods(userType,
                    (ReflectionUtils.MethodFilter) method -> Objects.nonNull(AnnotatedElementUtils.findMergedAnnotation(method, RequestMapping.class)));
            for (Method method : methods) {
                final RequestMapping requestMapping = AnnotatedElementUtils.findMergedAnnotation(method, RequestMapping.class);
                builderHandlerMappingAndRegister(userType, method, beanName, getMappingForMethod(beanClass, requestMapping));
            }
        }
    }


    private RequestMappingInfo getMappingForMethod(Class<?> beanClass, RequestMapping requestMapping) {
        RequestMappingInfo info = createRequestMappingInfo(requestMapping);
        if (info != null) {
            final RequestMapping ctrlMapping = AnnotatedElementUtils.findMergedAnnotation(beanClass, RequestMapping.class);
            RequestMappingInfo typeInfo = createRequestMappingInfo(ctrlMapping);
            if (typeInfo != null) {
                info = typeInfo.combine(info);
            }
            info = RequestMappingInfo.paths("/").build().combine(info);
        }
        return info;
    }


    protected RequestMappingInfo createRequestMappingInfo(
            RequestMapping requestMapping) {
        if (Objects.isNull(requestMapping)){
            return null;
        }
        return RequestMappingInfo
                .paths(requestMapping.path())
                .methods(requestMapping.method())
                .mappingName(requestMapping.name())
                .build();
    }
}
