package cn.com.duiba.linglong.client.cluster.hazelcast;

import cn.com.duiba.linglong.client.cluster.ClusterProperties;
import com.hazelcast.client.config.ClientConfig;
import com.hazelcast.client.config.ClientNetworkConfig;
import com.hazelcast.config.DiscoveryStrategyConfig;
import com.hazelcast.core.HazelcastInstance;
import com.hazelcast.core.LifecycleService;
import com.hazelcast.cp.CPSubsystem;
import com.hazelcast.spi.discovery.DiscoveryStrategyFactory;
import com.hazelcast.spring.context.SpringManagedContext;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.hazelcast.HazelcastClientFactory;
import org.springframework.context.ApplicationContext;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

import javax.annotation.Resource;

/**
 * @author liuyao
 */
@Configuration
@Slf4j
public class HazelcastConfiguration {

    public static final String SPRING_APPLICATION_CONTEXT_KEY = "applicationContext";

    /**
     * 打开 @SpringAware 的功能
     */
    @Bean
    public SpringManagedContext springManagedContext(){
        return new SpringManagedContext();
    }

    @Bean
    public ZookeeperDiscoveryStrategy zookeeperDiscoveryStrategy(){
        return new ZookeeperDiscoveryStrategy();
    }

    @Bean
    public DiscoveryStrategyFactory discoveryStrategyFactory(){
        return new ZookeeperDiscoveryStrategyFactory();
    }

    @Bean
    public LifecycleService lifecycleService(@Qualifier("linglongHazelcastClient") HazelcastInstance hazelcastInstance){
        return hazelcastInstance.getLifecycleService();
    }

    @Bean
    public CPSubsystem cpSubsystem(@Qualifier("linglongHazelcastClient") HazelcastInstance hazelcastInstance){
        return hazelcastInstance.getCPSubsystem();
    }

    /**
     * 控制台只启动客户端对集群进行干预
     */
    @Configuration
    public static class HazelcastClientConfiguration{

        @Resource
        private ApplicationContext applicationContext;
        @Resource
        private ClusterProperties clusterProperties;

        @Bean
        public ClientConfig clientConfig(
                DiscoveryStrategyFactory discoveryStrategyFactory,
                SpringManagedContext springManagedContext
        ){

            ClientConfig clientConfig = new ClientConfig();
            clientConfig.setClusterName(clusterProperties.getGroup());
            clientConfig.setManagedContext(springManagedContext);
            clientConfig.getUserContext().put(SPRING_APPLICATION_CONTEXT_KEY, applicationContext);
            ClientNetworkConfig networkConfig = clientConfig.getNetworkConfig();
            networkConfig.getAwsConfig().setEnabled(false);
            networkConfig.setSmartRouting(true);
            networkConfig.setRedoOperation(true);

            clientConfig.setProperty("hazelcast.discovery.enabled", "true");
            clientConfig.setProperty("hazelcast.logging.type", "slf4j");
            clientConfig.setProperty("hazelcast.socket.client.bind.any", "false");

            clientConfig.setProperty("hazelcast.heartbeat.failuredetector.type", "deadline");
            clientConfig.setProperty("hazelcast.heartbeat.interval.seconds", "5");
            clientConfig.setProperty("hazelcast.max.no.heartbeat.seconds", "20");

            DiscoveryStrategyConfig strategyConfig = new DiscoveryStrategyConfig(discoveryStrategyFactory);
            networkConfig.getDiscoveryConfig().addDiscoveryStrategyConfig(strategyConfig);
            networkConfig
                    .setSmartRouting(true)
                    .addOutboundPortDefinition("34700-34710")
                    .setRedoOperation(true)
                    .setConnectionTimeout(5000);
            return clientConfig;
        }

        @Bean
        @ConditionalOnMissingBean(name = "linglongHazelcastClient")
        public HazelcastInstance linglongHazelcastClient(ClientConfig config) {
            return (new HazelcastClientFactory(config)).getHazelcastInstance();
        }
    }

}
