package cn.com.duiba.linglong.client.job.jobs;

import cn.com.duiba.linglong.client.domain.dto.JobKey;
import cn.com.duiba.linglong.client.domain.event.JobCancelEvent;
import cn.com.duiba.linglong.client.job.ScheduleProperties;
import cn.com.duiba.linglong.client.job.consumer.JobConsumerAssert;
import com.github.benmanes.caffeine.cache.Caffeine;
import com.github.benmanes.caffeine.cache.LoadingCache;
import com.github.benmanes.caffeine.cache.RemovalListener;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.springframework.context.ApplicationContext;
import org.springframework.context.event.EventListener;

import javax.annotation.Resource;
import java.util.List;
import java.util.Objects;

/**
 * @author liuyao
 */
@Slf4j
public class WorkerScheduleJobManager implements JobConsumerAssert {

    @Resource
    private ApplicationContext applicationContext;
    @Resource
    private ScheduleProperties scheduleProperties;

    private final LoadingCache<JobKey, WorkerJobRunnable> runnings = Caffeine.newBuilder()
                    .removalListener((RemovalListener<JobKey, WorkerJobRunnable>) (key, runnable, cause) -> {
                        assert runnable != null;
                        if(runnable.isRunning()){
                            runnable.stopAsync();
                        }
                    })
                    .build(jobKey -> {
                        WorkerJobRunnable runnable = applicationContext.getBean(WorkerJobRunnable.class);
                        runnable.setJobKey(jobKey);
                        return runnable;
                    });

    public synchronized void submitScheduleJob(JobKey jobKey){
        WorkerJobRunnable runnable = runnings.getIfPresent(jobKey);
        //保证幂等
        if(Objects.nonNull(runnable)){
            return;
        }
        WorkerJobRunnable newRunnable = runnings.get(jobKey);
        Objects.requireNonNull(newRunnable);
        newRunnable.startAsync();
    }

    @EventListener(JobCancelEvent.class)
    public synchronized void cancelJob(JobCancelEvent event){
        JobKey key = new JobKey(event.getInvokeType(),event.getHistoryId());
        WorkerJobRunnable runnable = runnings.getIfPresent(key);
        if(Objects.isNull(runnable) || !runnable.isRunning()){
            return;
        }
        runnable.cancel();
        runnings.invalidate(key);
    }

    public List<JobKey> findAllRunningJobs(){
        return Lists.newArrayList(runnings.asMap().keySet());
    }

    public synchronized void clearJob(JobKey jobKey){
        runnings.invalidate(jobKey);
    }

    @Override
    public boolean canConsumer() {
        return runnings.estimatedSize() < scheduleProperties.getMaxJobSize();
    }
}
