package cn.com.duiba.linglong.client.job.consumer;

import cn.com.duiba.boot.event.MainContextRefreshedEvent;
import cn.com.duiba.linglong.client.cluster.hazelcast.Node;
import cn.com.duiba.linglong.client.cluster.hazelcast.ZookeeperDiscoveryStrategy;
import cn.com.duiba.linglong.client.constant.JobInvokeType;
import cn.com.duiba.linglong.client.domain.dto.JobKey;
import cn.com.duiba.linglong.client.job.jobs.WorkerScheduleJobManager;
import cn.com.duiba.linglong.client.remoteservice.RemoteWorkerStatusService;
import cn.com.duiba.linglong.client.service.channel.JobConsumerChannel;
import cn.com.duiba.linglong.client.service.channel.JobConsumerChannelManager;
import cn.com.duiba.linglong.client.service.channel.JobInvoker;
import com.google.common.util.concurrent.AbstractExecutionThreadService;
import com.hazelcast.client.HazelcastClientNotActiveException;
import com.hazelcast.core.HazelcastInstanceNotActiveException;
import lombok.extern.slf4j.Slf4j;
import org.apache.curator.x.discovery.ServiceInstance;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.event.EventListener;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import javax.annotation.Resource;
import java.util.List;
import java.util.Objects;

/**
 * 任务消费者
 * @author liuyao
 */
@Slf4j
public class WorkerJobConsumer extends AbstractExecutionThreadService {

    @Resource
    private List<JobConsumerAssert> jobConsumerAsserts;
    @Resource
    private JobConsumerChannelManager jobConsumerChannelManager;
    @Resource
    private WorkerScheduleJobManager workerScheduleJobManager;
    @Resource
    private ZookeeperDiscoveryStrategy zookeeperDiscoveryStrategy;
    @Resource
    private RemoteWorkerStatusService remoteWorkerStatusService;
    @Value("${spring.application.name}")
    private String appName;

    private JobConsumerChannel jobConsumerChannel;

    private Thread heartbeat;

    @EventListener(MainContextRefreshedEvent.class)
    public void startConsumer(){
        jobConsumerChannel = jobConsumerChannelManager.getJobConsumerChannel(appName);
        this.startAsync();
        WorkerJobConsumer consumer = this;
        //noinspection AlibabaAvoidManuallyCreateThread
        heartbeat = new Thread(() -> {
            consumer.awaitRunning();
            while (isRunning()){
                try {
                    ServiceInstance<Node> instance = zookeeperDiscoveryStrategy.getThisServiceInstance();
                    remoteWorkerStatusService.heartbeat(instance.getName(),instance.getId());
                    Thread.sleep(5000L);
                }catch (InterruptedException e) {
                    break;
                }catch (Exception e){
                    log.error("心跳上报失败",e);
                }
            }
        });
        heartbeat.setDaemon(true);
        heartbeat.setName("linglong.client.heartbeat");
        heartbeat.setPriority(Thread.MAX_PRIORITY);
        heartbeat.start();
    }

    @PreDestroy
    public void preDestroy(){
        if(isRunning()){
            this.stopAsync();
        }
    }

    @Override
    protected void triggerShutdown() {
        //终断心跳
        heartbeat.interrupt();
        ServiceInstance<Node> instance = zookeeperDiscoveryStrategy.getThisServiceInstance();
        remoteWorkerStatusService.stopWorker(instance.getName(),instance.getId());
    }

    @Override
    protected void startUp() {
        log.info("Worker任务消费通道构建完成");
    }

    @Override
    public void run(){
        while (isRunning()){
            try {
                if(!canRunJob()){
                    int retryTime = 3;
                    for(int i = 0; i< retryTime; i++){
                        if(!isRunning()){
                            return;
                        }
                        Thread.sleep(1000);
                    }
                    continue;
                }
                JobInvoker invoker = jobConsumerChannel.takeJob();
                if(Objects.isNull(invoker)){
                    Thread.sleep(1000);
                    if(!isRunning()){
                        return;
                    }
                    continue;
                }
                try{
                    JobKey jobKey = new JobKey(JobInvokeType.ACTION,invoker.getHistoryId());
                    workerScheduleJobManager.submitScheduleJob(jobKey,invoker.getJobLevel());
                }catch (Exception e){
                    log.error("任务运行失败",e);
                }
            } catch (InterruptedException| HazelcastInstanceNotActiveException| HazelcastClientNotActiveException e) {
                break;
            }catch (Exception e){
                log.error("任务消费管道异常",e);
            }
        }
        if(!Thread.currentThread().isInterrupted()){
            Thread.currentThread().interrupt();
        }
    }

    private boolean canRunJob() {
        for(JobConsumerAssert consumerAssert:jobConsumerAsserts){
            if(!consumerAssert.canConsumer()){
                return false;
            }
        }
        return true;
    }

}
