package cn.com.duiba.linglong.client.job.jobs;

import cn.com.duiba.boot.exception.BizException;
import cn.com.duiba.linglong.client.domain.dto.JobInvokerInfoDto;
import cn.com.duiba.linglong.client.domain.dto.JobKey;
import cn.com.duiba.linglong.client.domain.event.JobCancelEvent;
import cn.com.duiba.linglong.client.domain.params.JobCallback;
import cn.com.duiba.linglong.client.domain.params.JobRunningCallback;
import cn.com.duiba.linglong.client.job.WorkerScheduleProperties;
import cn.com.duiba.linglong.client.job.consumer.JobConsumerAssert;
import cn.com.duiba.linglong.client.remoteservice.RemoteActionCallbackService;
import cn.com.duiba.linglong.client.service.channel.JobLevel;
import com.github.benmanes.caffeine.cache.Caffeine;
import com.github.benmanes.caffeine.cache.LoadingCache;
import com.github.benmanes.caffeine.cache.RemovalListener;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.springframework.context.ApplicationContext;
import org.springframework.context.event.EventListener;

import javax.annotation.Resource;
import java.util.List;
import java.util.Objects;

/**
 * @author liuyao
 */
@Slf4j
public class WorkerScheduleJobManager implements JobConsumerAssert {

    private static final int MAX_ERROR_COUNT = 6;

    @Resource
    private ApplicationContext applicationContext;
    @Resource
    private WorkerScheduleProperties scheduleProperties;
    @Resource
    private RemoteActionCallbackService remoteActionCallbackService;

    private final LoadingCache<JobKey, WorkerJobRunnable> runnings = Caffeine.newBuilder()
                    .removalListener((RemovalListener<JobKey, WorkerJobRunnable>) (key, runnable, cause) -> {
                        assert runnable != null;
                        if(runnable.isRunning()){
                            runnable.stopAsync();
                        }
                    })
                    .build(jobKey -> {
                        WorkerJobRunnable runnable = applicationContext.getBean(WorkerJobRunnable.class);
                        runnable.setJobKey(jobKey);
                        return runnable;
                    });

    public synchronized void submitScheduleJob(JobKey jobKey, JobLevel jobLevel){
        WorkerJobRunnable runnable = runnings.getIfPresent(jobKey);
        //保证幂等
        if(Objects.nonNull(runnable)){
            return;
        }
        WorkerJobRunnable newRunnable = runnings.get(jobKey);
        Objects.requireNonNull(newRunnable);
        newRunnable.setJobLevel(jobLevel);
        newRunnable.startAsync();
    }

    /**
     * 任务运行回调
     */
    public JobInvokerInfoDto runningAck(JobRunningCallback callback) throws BizException {
        WorkerJobRunnable runnable = runnings.getIfPresent(callback.getJobKey());
        if(Objects.isNull(runnable)){
            throw new RuntimeException("任务未注册");
        }
        int errorCount = 0;
        while (runnable.isRunning() && errorCount < MAX_ERROR_COUNT){
            try{
                return remoteActionCallbackService.runningAck(callback);
            }catch (BizException e){
                throw e;
            }catch (Exception e){
                log.error("任务运行回调失败",e);
                errorCount ++;
                try{
                    Thread.sleep(30000L);
                }catch (InterruptedException interruptedException){
                    log.error("重试等待中断",interruptedException);
                }
            }
        }
        throw new BizException("任务运行回调失败");
    }

    /**
     * 任务执行完成回调
     */
    public void jobCallback(JobCallback callback) throws BizException{
        WorkerJobRunnable runnable = runnings.getIfPresent(callback.getJobKey());
        if(Objects.isNull(runnable) || !runnable.isRunning()){
            return;
        }
        int errorCount = 0;
        while (runnable.isRunning() && errorCount < MAX_ERROR_COUNT){
            try{
                remoteActionCallbackService.jobCallback(callback);
                return;
            }catch (BizException e){
                throw e;
            }catch (Exception e){
                log.error("任务运行回调失败",e);
                errorCount ++;
                try{
                    Thread.sleep(30000L);
                }catch (InterruptedException interruptedException){
                    log.error("重试等待中断",interruptedException);
                }
            }
        }
    }

    @EventListener(JobCancelEvent.class)
    public synchronized void cancelJob(JobCancelEvent event){
        JobKey key = new JobKey(event.getInvokeType(),event.getHistoryId());
        WorkerJobRunnable runnable = runnings.getIfPresent(key);
        if(Objects.isNull(runnable) || !runnable.isRunning()){
            return;
        }
        runnable.cancel();
        runnings.invalidate(key);
    }

    public List<JobKey> findAllRunningJobs(){
        return Lists.newArrayList(runnings.asMap().keySet());
    }

    public void clearJob(JobKey jobKey){
        runnings.invalidate(jobKey);
    }

    @Override
    public boolean canConsumer() {
        return runnings.estimatedSize() < scheduleProperties.getMaxJobSize();
    }
}
