package cn.com.duiba.linglong.client.job.interceptor;

import cn.com.duiba.linglong.client.exception.WorkerException;
import cn.com.duiba.linglong.client.job.jobs.AbstractJob;
import cn.com.duiba.linglong.client.job.jobs.JobContext;
import lombok.extern.slf4j.Slf4j;

import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;

/**
 * @author liuyao
 */
@Slf4j
public class WithProcesserJob extends AbstractJob {

	private AbstractJob job;

	private List<JobInterceptor> interceptors = Collections.emptyList();

	private int interceptorIndex = -1;

	public void setInterceptors(List<JobInterceptor> interceptors) {
		this.interceptors = interceptors;
	}

	public void setJob(AbstractJob job) {
		this.job = job;
	}

	@Override
	public void run() throws WorkerException {
		if(Objects.isNull(job)){
			return;
		}
		try {
			if (applyPreHandle(job) && job.isRunning()) {
				job.run();
			}
		} finally {
			applyPostHandle(job);
		}
	}

	@Override
	protected void cancelInternal() {
		if(Objects.nonNull(job) && job.isRunning()){
			job.cancel();
		}
	}

	@Override
	public JobContext getJobContext() {
		return Optional.ofNullable(job).map(AbstractJob::getJobContext).orElse(null);
	}

	@Override
	public boolean isRunning() {
		if(Objects.isNull(job)){
			return false;
		}
		return job.isRunning();
	}

	/**
	 * Apply preHandle methods of registered interceptors.
	 */
	private boolean applyPreHandle(AbstractJob job) throws WorkerException{
		if (interceptors.size()!=0) {
			for (int i = 0; i < interceptors.size(); i++) {
				JobInterceptor interceptor = interceptors.get(i);
				if (!interceptor.preHandle(job)) {
					return false;
				}
				this.interceptorIndex = i;
			}
		}
		return true;
	}

	/**
	 * Apply postHandle methods of registered interceptors.
	 */
	private void applyPostHandle(AbstractJob job){

		if (interceptors.size()!=0) {
			for (int i = this.interceptorIndex; i >= 0; i--) {
				JobInterceptor interceptor = interceptors.get(i);
				try {
					interceptor.postHandle(job);
				}catch (Throwable ex2) {
					log.error("HandlerInterceptor.afterCompletion threw exception", ex2);
				}
			}
		}
	}


}
