package cn.com.duiba.linglong.sdk.cluster.hazelcast;

import cn.com.duiba.boot.utils.NetUtils;
import cn.com.duiba.linglong.sdk.cluster.zookeeper.ZookeeperProperties;
import cn.com.duiba.linglong.sdk.utils.IOUtils;
import com.google.common.collect.Lists;
import com.hazelcast.cluster.Address;
import com.hazelcast.spi.discovery.DiscoveryNode;
import com.hazelcast.spi.discovery.DiscoveryStrategy;
import com.hazelcast.spi.discovery.SimpleDiscoveryNode;
import com.hazelcast.spi.partitiongroup.PartitionGroupStrategy;
import lombok.extern.slf4j.Slf4j;
import org.apache.curator.framework.CuratorFramework;
import org.apache.curator.x.discovery.ServiceDiscovery;
import org.apache.curator.x.discovery.ServiceDiscoveryBuilder;
import org.apache.curator.x.discovery.ServiceInstance;
import org.apache.curator.x.discovery.UriSpec;
import org.springframework.beans.factory.annotation.Value;

import javax.annotation.Resource;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;

/**
 * @author liuyao
 */
@Slf4j
public class ZookeeperDiscoveryStrategy implements DiscoveryStrategy {

    @Resource
    private CuratorFramework curatorFramework;
    @Resource
    private ZookeeperProperties canalZookeeperProperties;
    @Value("${spring.application.name}")
    private String appName;
    @Value("${server.port}")
    private int httpServerPort;

    private DiscoveryNode discoveryNode;

    private ServiceDiscovery<Node> serviceDiscovery;
    private ServiceInstance<Node> serviceInstance;

    void setDiscoveryNode(DiscoveryNode discoveryNode) {
        this.discoveryNode = discoveryNode;
    }

    private boolean isMember() {
        return discoveryNode != null;
    }

    public ServiceInstance<Node> getThisServiceInstance() {
        return serviceInstance;
    }

    public ServiceDiscovery<Node> getServiceDiscovery() {
        return serviceDiscovery;
    }

    @Override
    public void start() {
        try {
            ServiceDiscoveryBuilder<Node> discoveryBuilder = ServiceDiscoveryBuilder.builder(Node.class)
                    .basePath(canalZookeeperProperties.getRootPath())
                    .client(curatorFramework);
            //register members only into zookeeper
            //there no need to register clients

            //register members only into zookeeper
            //there no need to register clients
            prepareServiceInstance();
            discoveryBuilder.thisInstance(serviceInstance);

            serviceDiscovery = discoveryBuilder.build();
            serviceDiscovery.start();
        } catch (Exception e) {
            throw new IllegalStateException("Error while talking to ZooKeeper. ", e);
        }
    }

    private void prepareServiceInstance() throws Exception {

        Node node = new Node();
        String address = NetUtils.getLocalIp();
        if (isMember()) {
            Address privateAddress = discoveryNode.getPrivateAddress();
            address = privateAddress.getHost();
            node.setHzPort(privateAddress.getPort());
        }
        node.setWeight(100);
        serviceInstance = ServiceInstance.<Node>builder()
                .id(address+":"+httpServerPort)
                .uriSpec(new UriSpec("{scheme}://{address}:{port}"))
                .address(address)
                .port(httpServerPort)
                .name(appName)
                .payload(node)
                .build();
    }

    @Override
    public Iterable<DiscoveryNode> discoverNodes() {
        List<DiscoveryNode> nodes = Lists.newArrayList();
        try {
            //以server作为服务节点
            Collection<ServiceInstance<Node>> instances = serviceDiscovery.queryForInstances("linglong");
            for (ServiceInstance<Node> serviceInstance : instances) {
                String host = serviceInstance.getAddress();
                //取hazelcast的端口号
                int port = serviceInstance.getPayload().getHzPort();
                Address address = new Address(host, port);
                SimpleDiscoveryNode node = new SimpleDiscoveryNode(address);
                nodes.add(node);
            }
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new IllegalStateException("Error while talking to ZooKeeper", e);
        } catch (Exception e) {
            throw new IllegalStateException("Error while talking to ZooKeeper", e);
        }
        return nodes;
    }

    public Collection<ServiceInstance<Node>> members(String hostGroup){
        try{
            return serviceDiscovery.queryForInstances(hostGroup);
        }catch (Exception e) {
            throw new IllegalStateException("Error while talking to ZooKeeper", e);
        }
    }

    public ServiceInstance<Node> findServiceInstanceByMember(String hostGroup, String address){
        try{
            return serviceDiscovery.queryForInstance(hostGroup,address);
        }catch (Exception e) {
            throw new IllegalStateException("Error while talking to ZooKeeper", e);
        }
    }

    @Override
    public void destroy() {
        try {
            if (isMember() && serviceDiscovery != null) {
                serviceDiscovery.unregisterService(serviceInstance);
            }
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new IllegalStateException("Error while talking to ZooKeeper", e);
        } catch (Exception e) {
            throw new IllegalStateException("Error while talking to ZooKeeper", e);
        } finally {
            IOUtils.closeSafely(serviceDiscovery);
        }
    }

    @Override
    public PartitionGroupStrategy getPartitionGroupStrategy() {
        return null;
    }

    @Override
    public Map<String, String> discoverLocalMetadata() {
        return Collections.emptyMap();
    }
}
