package cn.com.duiba.linglong.sdk.monitor;

import cn.com.duiba.linglong.sdk.job.consumer.JobConsumerAssert;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Component;

import javax.annotation.PostConstruct;
import javax.annotation.Resource;
import java.io.BufferedReader;
import java.io.InputStreamReader;
import java.util.List;

/**
 * @author liuyao
 */
@Slf4j
@Component
public class CpuInfoService implements JobConsumerAssert {

    private static final List<String> LOAD_PREFIX = Lists.newArrayList("load average:","load averages:");

    private volatile String loadPrefix;

    private final List<String> commands = Lists.newArrayList("sh","-c","uptime");

    @Resource
    private MonitorProperties monitorProperties;

    private final Integer cpuCoreNum = Runtime.getRuntime().availableProcessors();

    private ProcessBuilder processBuilder;

    private volatile Float cpuloadPercore = 0f;

    @PostConstruct
    public void init(){
        processBuilder = new ProcessBuilder(commands);
        processBuilder.redirectErrorStream(true);
        String uptime = uptime();
        for(String prefix:LOAD_PREFIX){
            if(uptime.contains(prefix)){
                loadPrefix = prefix;
                break;
            }
        }
    }

    private String uptime(){
        try{
            Process process = processBuilder.start();
            StringBuilder stringBuilder = new StringBuilder();
            String line;
            try(BufferedReader br = new BufferedReader(new InputStreamReader(process.getInputStream()))){
                while ((line = br.readLine()) != null) {
                    stringBuilder.append(line);
                }
            }catch (Exception e){
                log.debug("shell执行结果监听失败",e);
            }
            int exit = process.waitFor();
            if(exit != 0){
                log.error("uptime error:{}",stringBuilder);
                return "";
            }
            return stringBuilder.toString();
        }catch (Exception e){
            log.error("upTime执行失败",e);
        }
        return "";
    }

    @Scheduled(cron = "0/3 * * * * ?")
    public void runOneIteration() {
        if(StringUtils.isBlank(loadPrefix)){
            return;
        }
        String uptime = uptime();
        if(StringUtils.isBlank(uptime)){
            return;
        }
        int index = uptime.indexOf(loadPrefix);
        String sub = uptime.substring(index).substring(loadPrefix.length()).trim();
        String[] tmps = sub.split(",");
        if(tmps.length==1){
            tmps = sub.split(" ");
        }
        //最近1分钟系统的平均cpu负载
        Float cpuload = Float.valueOf(tmps[0].trim());
        this.cpuloadPercore = cpuload / cpuCoreNum;
    }

    @Override
    public boolean canConsumer() {
        if(cpuloadPercore > monitorProperties.getMaxCpuLoadPerCore()){
            log.info("CPU过载：max("+monitorProperties.getMaxCpuLoadPerCore()+"),current:("+cpuloadPercore+")");
            return false;
        }
        return true;
    }
}
