package cn.com.duiba.linglong.sdk.service.channel;

import cn.com.duiba.application.boot.api.domain.dto.ApplicationDto;
import cn.com.duiba.application.boot.api.remoteservice.RemoteApplicationService;
import com.github.benmanes.caffeine.cache.CacheLoader;
import com.github.benmanes.caffeine.cache.Caffeine;
import com.github.benmanes.caffeine.cache.LoadingCache;
import org.checkerframework.checker.nullness.qual.NonNull;
import org.jetbrains.annotations.NotNull;
import org.springframework.context.ApplicationContext;

import javax.annotation.Resource;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.TimeUnit;

/**
 * @author liuyao
 */
public class JobConsumerChannelManager {

    @Resource
    private ApplicationContext applicationContext;
    @Resource
    private RemoteApplicationService remoteApplicationService;

    private final LoadingCache<String,JobConsumerChannel> consumerChannelMap = Caffeine.newBuilder().expireAfterAccess(1, TimeUnit.HOURS).build(new CacheLoader<String, JobConsumerChannel>() {
        @NotNull
        @Override
        public JobConsumerChannel load(@NonNull String appName) {
            ApplicationDto application = remoteApplicationService.getApplicationByAlias(appName);
            if(Objects.isNull(application) || application.getAbandoned()){
                throw new RuntimeException("无效的应用:"+appName);
            }
            JobConsumerChannel channel = applicationContext.getBean(JobConsumerChannel.class);
            channel.init(appName);
            return channel;
        }
    });

    public void acceptJob(JobInvoker invoker){
        Objects.requireNonNull(invoker.getAppName());
        JobConsumerChannel channel = consumerChannelMap.get(invoker.getAppName());
        Objects.requireNonNull(channel);
        JobLevel level = Optional.ofNullable(invoker.getJobLevel()).orElse(JobLevel.MIDDLE);
        channel.pushJobKey(level,invoker.getHistoryId());
    }

    public void removeJob(String appName,Long historyId){
        JobConsumerChannel channel = consumerChannelMap.get(appName);
        Objects.requireNonNull(channel);
        channel.removeJobKey(historyId);
    }

    public JobConsumerChannel getJobConsumerChannel(String appName){
        return consumerChannelMap.get(appName);
    }

}
