package cn.com.duiba.linglong.sdk.service.logger;

import cn.com.duiba.linglong.sdk.domain.dto.JobKey;
import cn.com.duiba.wolf.utils.UUIDUtils;
import com.google.common.collect.Lists;
import com.hazelcast.collection.ISet;
import com.hazelcast.core.HazelcastInstance;
import com.hazelcast.map.IMap;
import com.hazelcast.multimap.MultiMap;

import javax.annotation.PostConstruct;
import javax.annotation.Resource;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.TimeUnit;

/**
 * @author liuyao
 */
public class JobLogService {

    @Resource
    private HazelcastInstance hazelcastInstance;
    @Resource
    private JobLogSessionService jobLogSessionService;

    private ISet<String> logKeySet;

    private MultiMap<String,JobLog> jobLogMultiMap;
    /**
     * 日志转存后，如果是存到ES里，数据存入后因为是异步的提交方式，索引的查询存在延时，在延时的时间窗口，日志会查询不到
     * 引入delayLogMap，在日志被转存后再保留一段时间
     * "index.refresh_interval" : “5s,
     * "index.translog.durability" : "async",
     */
    private IMap<String,List<JobLog>> delayLogMap;

    @PostConstruct
    public void init(){
        logKeySet = hazelcastInstance.getSet("JobLogService.logKeySet");
        jobLogMultiMap = hazelcastInstance.getMultiMap("JobLogService.jobLogMultiMap");
        delayLogMap = hazelcastInstance.getMap("JobLogService.delayLogMap");
    }

    public void startLog(JobKey jobKey){
        logKeySet.add(jobKey.key());
    }

    public void acceptLog(JobKey jobKey,JobLog jobLog) {
        String key = jobKey.key();
        if(!logKeySet.contains(key)){
            return;
        }
        jobLog.setUuid(UUIDUtils.createUUID());
        jobLogMultiMap.put(key,jobLog);

        jobLogSessionService.acceptLog(jobKey,jobLog);
    }

    public Collection<JobLog> releaseLog(JobKey jobKey){
        String key = jobKey.key();
        logKeySet.remove(key);
        Collection<JobLog> logs = jobLogMultiMap.remove(key);
        if(!logs.isEmpty()){
            delayLogMap.put(key, Lists.newArrayList(logs),5, TimeUnit.SECONDS);
        }
        return logs;
    }

    public List<JobLog> findLogsByJobKey(JobKey jobKey){
        String key = jobKey.key();
        if(logKeySet.contains(key)){
            return Lists.newArrayList(jobLogMultiMap.get(key));
        }
        if(delayLogMap.containsKey(key)){
            return delayLogMap.get(key);
        }
        return Collections.emptyList();
    }

}
