package cn.com.duiba.message.service.rest.controller;

import cn.com.duiba.message.service.api.remoteservice.RemoteLetterService;
import com.github.benmanes.caffeine.cache.Cache;
import com.github.benmanes.caffeine.cache.Caffeine;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.springframework.scheduling.annotation.Async;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Component;
import org.springframework.web.servlet.mvc.method.annotation.SseEmitter;

import javax.annotation.Resource;
import java.io.IOException;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;

/**
 * @author: <a href="http://www.panaihua.com">panaihua</a>
 * @date: 2019-10-17 16:01
 * @descript:
 * @version: 1.0
 */
@Component
@Slf4j
public class SsePushNotifySchedule {

    @Resource
    private RemoteLetterService remoteLetterService;

    private final Map<String, SseEmitter> emitters = new ConcurrentHashMap<>();

    /**
     * 10分钟内不会在提交
     */
    private final Cache<String, Boolean> alreadySend = Caffeine.newBuilder().expireAfterWrite(10, TimeUnit.MINUTES).build();

    void registerEmitter(final String receiveId, final SseEmitter emitter) {

        emitter.onCompletion(() -> this.removeEmitter(receiveId));
        emitter.onTimeout(() -> this.removeEmitter(receiveId));
        // 如果不发送在域名访问下会断开连接（ip访问下没有问题），所以第一次连接时发送一个消息
        SseEmitter.SseEventBuilder builder = SseEmitter.event()
                .id(UUID.randomUUID().toString()).data("connected");
        try {
            emitter.send(builder);
        } catch (IOException e) {
            return;
        }

        emitters.put(receiveId, emitter);
    }

    boolean isExist(String receiveId) {
        return emitters.containsKey(receiveId);
    }

    private void removeEmitter(final String receiveId) {
        emitters.remove(receiveId);
    }

    @Scheduled(fixedRate = 10000)
    public void doNotify() {

        List<String> deadEmitters = Lists.newArrayList();
        List<String> receiveIds = remoteLetterService.hasNewLetter(emitters.keySet());
        for (Map.Entry<String, SseEmitter> entry : emitters.entrySet()) {

            SseEmitter sseEmitter = entry.getValue();
            String receiveId = entry.getKey();
            if(!receiveIds.contains(receiveId) || alreadySend.getIfPresent(receiveId) != null) {
                continue;
            }

            SseEmitter.SseEventBuilder builder = SseEmitter.event()
                    .id(UUID.randomUUID().toString()).data(true);
            try {
                sseEmitter.send(builder);
                alreadySend.put(receiveId, Boolean.TRUE);
                sseEmitter.complete();
            } catch (IOException e) {
                deadEmitters.add(receiveId);
            }
        }

        deadEmitters.forEach(emitters::remove);
    }

}
