package cn.com.duiba.quanyi.center.api.enums.activity.common;

import cn.com.duiba.quanyi.center.api.dto.common.ActivityPeriodicDto;
import cn.com.duiba.wolf.utils.DateUtils;
import cn.hutool.core.date.DatePattern;
import cn.hutool.core.date.DateUtil;
import lombok.AllArgsConstructor;
import lombok.Data;
import lombok.Getter;

import java.util.Arrays;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * @author lizhi
 * @date 2025/4/22 14:00
 */
@AllArgsConstructor
@Getter
public enum ActivityPeriodicUnitEnum {
    FOREVER(1, "永久"),
    ABSOLUTE(2, "绝对时间"),
    DAY(3, "日"),
    WEEK(4, "自然周"),
    MONTH(5, "自然月"),
    QUARTER(6,"季度"),
    ;

    private final Integer code;

    private final String desc;

    private final static Date FOREVER_DATE_BEGIN = DateUtils.getDayDate("2000-01-01");
    private final static Date FOREVER_DATE_END = DateUtils.getDayDate("2099-01-01");
    /**
     * 获取当前周期范围
     * @param now
     * @param absoluteStartTime
     * @param absoluteEndTime
     * @return
     */
    public ActivityPeriodicDto getDate(Date now, Date absoluteStartTime, Date absoluteEndTime) {
        switch (this) {
            case FOREVER:
                return new ActivityPeriodicDto("20000101", FOREVER_DATE_BEGIN, FOREVER_DATE_END);
            case ABSOLUTE:
                return new ActivityPeriodicDto(DatePattern.PURE_DATE_FORMAT.format(absoluteStartTime), absoluteStartTime, absoluteEndTime);
            case DAY:
                return new ActivityPeriodicDto(DatePattern.PURE_DATE_FORMAT.format(DateUtil.beginOfDay(now)), DateUtil.beginOfDay(now), DateUtil.endOfDay(now));
            case WEEK:
                return new ActivityPeriodicDto(DatePattern.PURE_DATE_FORMAT.format(DateUtil.beginOfWeek(now)), DateUtil.beginOfWeek(now), DateUtil.endOfWeek(now));
            case MONTH:
                return new ActivityPeriodicDto(DatePattern.PURE_DATE_FORMAT.format(DateUtil.beginOfMonth(now)), DateUtil.beginOfMonth(now), DateUtil.endOfMonth(now));
            case QUARTER:
                return new ActivityPeriodicDto(DatePattern.PURE_DATE_FORMAT.format(DateUtil.beginOfQuarter(now)), DateUtil.beginOfQuarter(now), DateUtil.endOfQuarter(now));
            default:
                return null;
        }
    }

    /**
     * 获取当前期次值
     * @return
     */
    public ActivityPeriodicDto getCurrentPeriod() {
        return getDate(new Date(), null, null);
    }

    /**
     * 获取期次值
     * @param offset
     * @return
     */
    public ActivityPeriodicDto getOffsetPeriod(int offset) {
        Date now = new Date();
        switch (this) {
            case DAY:
                now = DateUtil.offsetDay(now, offset);
                break;
            case WEEK:
                now = DateUtil.offsetWeek(now, offset);
                break;
            case MONTH:
                now = DateUtil.offsetMonth(now, offset);
                break;
            case QUARTER:
                now = DateUtil.offsetMonth(now, offset * 3);
                break;
            default:
        }
        return getDate(now, null, null);
    }

    /**
     * 获取期次发放枚举集合
     * @return
     */
    public static List<ActivityPeriodicUnitEnum> getPeriodTakeEnums(){
        return Arrays.asList(ActivityPeriodicUnitEnum.DAY, ActivityPeriodicUnitEnum.WEEK, ActivityPeriodicUnitEnum.MONTH, ActivityPeriodicUnitEnum.QUARTER);
    }

    private static final Map<Integer, ActivityPeriodicUnitEnum> ENUM_MAP = Collections.unmodifiableMap(Arrays.stream(values()).collect(Collectors.toMap(ActivityPeriodicUnitEnum::getCode, Function.identity(), (v1, v2) -> v2)));

    /**
     * 根据枚举值获取枚举
     * @param code 枚举值
     * @return 枚举
     */
    public static ActivityPeriodicUnitEnum getByCode(Integer code) {
        if (code == null) {
            return null;
        }
        return ENUM_MAP.get(code);
    }
}