package cn.com.duiba.quanyi.center.api.remoteservice.demand;

import cn.com.duiba.boot.netflix.feign.AdvancedFeignClient;
import cn.com.duiba.quanyi.center.api.dto.demand.DemandChangeRecordDto;

import java.util.List;

/**
 * 需求变更记录表
 *
 * @author xuzhigang
 * @date 2025-07-07
 */
@AdvancedFeignClient
public interface RemoteDemandChangeRecordService {

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 需求变更记录
     */
    DemandChangeRecordDto selectById(Long id);

    /**
     * 根据需求ID查询变更记录列表
     *
     * @param demandId 需求ID
     * @return 变更记录列表
     */
    List<DemandChangeRecordDto> selectByDemandId(Long demandId);

    /**
     * 根据需求ID和变更类型查询变更记录列表
     *
     * @param demandId 需求ID
     * @param changeType 变更类型
     * @return 变更记录列表
     */
    List<DemandChangeRecordDto> selectByDemandIdAndChangeType(Long demandId, Integer changeType);

    /**
     * 根据审批状态查询变更记录列表
     *
     * @param approvalStatus 审批状态
     * @return 变更记录列表
     */
    List<DemandChangeRecordDto> selectByApprovalStatus(Integer approvalStatus);

    /**
     * 新增需求变更记录
     *
     * @param dto 需求变更记录
     * @return 新增记录的主键ID
     */
    Long insert(DemandChangeRecordDto dto);

    /**
     * 更新需求变更记录
     *
     * @param dto 需求变更记录
     * @return 影响行数
     */
    int update(DemandChangeRecordDto dto);

    /**
     * 根据主键删除需求变更记录
     *
     * @param id 主键
     * @return 影响行数
     */
    int deleteById(Long id);

    /**
     * 批量插入需求变更记录
     *
     * @param dtos 需求变更记录列表
     * @return 影响行数
     */
    int batchInsert(List<DemandChangeRecordDto> dtos);

    /**
     * 更新审批状态
     *
     * @param id 主键
     * @param approvalStatus 审批状态
     * @param errorMsg 失败原因
     * @return 影响行数
     */
    int updateApprovalStatus(Long id, Integer approvalStatus, String errorMsg);

    /**
     * 创建需求变更记录
     *
     * @param demandId 需求ID
     * @param changeType 变更类型
     * @param operatorSsoId 操作人SSO ID
     * @param operatorSsoName 操作人SSO姓名
     * @param changeReason 变更原因
     * @param credentialFiles 凭证文件
     * @param changeRecordDetail 变更记录明细
     * @param changeParam 变更参数JSON
     * @return 新增记录的主键ID
     */
    Long createChangeRecord(Long demandId, Integer changeType,
                           Long operatorSsoId, String operatorSsoName, String changeReason,
                           String credentialFiles, String changeRecordDetail, String changeParam);

    /**
     * 根据需求ID和多个变更类型查询变更记录列表
     *
     * @param demandId 需求ID
     * @param changeTypes 变更类型列表
     * @return 变更记录列表
     */
    List<DemandChangeRecordDto> selectByDemandIdAndChangeTypes(Long demandId, List<Integer> changeTypes);

    /**
     * 根据需求ID和多个变更类型查询新近一条变更记录
     *
     * @param demandId 需求ID
     * @param changeTypes 变更类型列表
     * @return 变更记录列表
     */
    DemandChangeRecordDto selectLatestByDemandIdAndChangeTypes(Long demandId, List<Integer> changeTypes);

    /**
     * 根据需求ID集合批量查询最近一次创建需求的审批状态
     *
     * @param demandIds 需求ID集合
     * @return 需求变更记录列表，每个需求ID对应最近一次创建需求的记录
     */
    List<DemandChangeRecordDto> selectLatestCreateDemandByDemandIds(List<Long> demandIds);

    /**
     * 根据需求ID查询最近一次创建需求的审批状态
     *
     * @param demandId 需求ID
     * @return 需求变更记录
     */
    DemandChangeRecordDto selectLatestCreateDemandByDemandId(Long demandId);
}
