package cn.com.duiba.quanyi.center.api.enums.activity;

import cn.com.duiba.quanyi.center.api.enums.goods.GoodsOrderStatusEnum;
import com.alibaba.fastjson.JSON;
import lombok.AllArgsConstructor;
import lombok.Getter;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;

import javax.annotation.Nonnull;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

/**
 * @author lizhi
 * @date 2024/4/1 16:32
 */
@Slf4j
@Getter
@AllArgsConstructor
public enum BatchTakeStatusEnum {
    INIT(1, "待发放"),
    PROCESSING(2, "发放中"),
    PART_FAIL(3, "部分发放失败"),
    ALL_FAIL(4, "全部发放失败"),
    ALL_SUCCESS(5, "全部成功"),
    SHORT_URL_INIT(6, "全部待领取"),
    SHORT_URL_PROCESSING(7, "领取中"),
    SHORT_URL_PART_FAIL(8, "部分领取失败"),
    SHORT_URL_ALL_FAIL(9, "全部领取失败"),
    OBJECT_PART_SUCCESS(10, "部分实物状态(可重试)"),
    OBJECT_SUCCESS(11, "全部实物状态(不可重试)"),
    ;
    
    private final Integer code;
    
    private final String desc;
    
    private static final Set<Integer> SHORT_URL_STATUS = Collections.unmodifiableSet(new HashSet<>(Arrays.asList(
            SHORT_URL_INIT.getCode(), 
            SHORT_URL_PROCESSING.getCode(), 
            SHORT_URL_PART_FAIL.getCode(), 
            SHORT_URL_ALL_FAIL.getCode())));
    
    public static boolean isShortUrlStatus(Integer status) {
        return SHORT_URL_STATUS.contains(status);
    }

    /**
     * 根据子订单状态，获取批量领取状态
     * @param goodsOrderStatusList 子订单状态集合
     * @return 批量状态
     */
    @Nonnull
    public static BatchTakeStatusEnum getByGoodsOrderStatus(List<Integer> goodsOrderStatusList) {
        if (CollectionUtils.isEmpty(goodsOrderStatusList)) {
            return BatchTakeStatusEnum.INIT;
        }
        int success = 0;
        int fail = 0;
        int process = 0;
        int notReceive = 0;
        int shortFail = 0;
        int shortProcess = 0;
        for (Integer goodsOrderStatus : goodsOrderStatusList) {
            if (GoodsOrderStatusEnum.PROCESS.getStatus().equals(goodsOrderStatus)) {
                process++;
            } else if (GoodsOrderStatusEnum.SHORT_URL_RECEIVE_PROCESS.getStatus().equals(goodsOrderStatus)) {
                shortProcess++;
            } else if (isFailGoodsOrderStatus(goodsOrderStatus)) {
                fail++;
            } else if (isShortFailGoodsOrderStatus(goodsOrderStatus)) {
                shortFail++;
            } else if (GoodsOrderStatusEnum.NOT_RECEIVE.getStatus().equals(goodsOrderStatus)) {
                notReceive++;
            } else {
                success++;
            }
        }
        if (success == goodsOrderStatusList.size()) {
            return BatchTakeStatusEnum.ALL_SUCCESS;
        }
        if (fail == goodsOrderStatusList.size()) {
            return BatchTakeStatusEnum.ALL_FAIL;
        }
        if (shortFail == goodsOrderStatusList.size()) {
            return BatchTakeStatusEnum.SHORT_URL_ALL_FAIL;
        }
        if (notReceive == goodsOrderStatusList.size()) {
            return BatchTakeStatusEnum.SHORT_URL_INIT;
        }
        if (fail + shortFail == goodsOrderStatusList.size()) {
            return BatchTakeStatusEnum.SHORT_URL_ALL_FAIL;
        }
        if (success + notReceive == goodsOrderStatusList.size()) {
            return BatchTakeStatusEnum.SHORT_URL_PROCESSING;
        }
        if (shortProcess > 0) {
            return BatchTakeStatusEnum.SHORT_URL_PROCESSING;
        }
        if (shortFail > 0) {
            return BatchTakeStatusEnum.SHORT_URL_PART_FAIL;
        }
        if (process > 0) {
            return BatchTakeStatusEnum.PROCESSING;
        }
        if (fail > 0) {
            return BatchTakeStatusEnum.PART_FAIL;
        }
        log.warn("[BatchTake], status error, goodsOrderStatusList={}", JSON.toJSONString(goodsOrderStatusList));
        return BatchTakeStatusEnum.PROCESSING;
    }

    private static boolean isFailGoodsOrderStatus(Integer goodsOrderStatus) {
        return GoodsOrderStatusEnum.FAIL.getStatus().equals(goodsOrderStatus)
                || GoodsOrderStatusEnum.EXCEPTION.getStatus().equals(goodsOrderStatus)
                || GoodsOrderStatusEnum.FAIL_CAN_RETRY.getStatus().equals(goodsOrderStatus);
    }

    private static boolean isShortFailGoodsOrderStatus(Integer goodsOrderStatus) {
        return GoodsOrderStatusEnum.SHORT_URL_RECEIVE_FAIL.getStatus().equals(goodsOrderStatus)
                || GoodsOrderStatusEnum.SHORT_URL_RECEIVE_FAIL_CAN_RETRY.getStatus().equals(goodsOrderStatus)
                || GoodsOrderStatusEnum.SHORT_URL_RECEIVE_EXCEPTION.getStatus().equals(goodsOrderStatus);
    }
}
