package cn.com.duiba.cloud.single.sign.on.contract.interceptor;


import cn.com.duiba.cloud.single.sign.on.contract.constants.SsoConstants;
import cn.com.duiba.cloud.single.sign.on.contract.interceptor.annotation.CanAccess;
import cn.com.duiba.cloud.single.sign.on.contract.tool.JsonRender;
import cn.com.duiba.cloud.single.sign.on.contract.tool.SsoRequestTool;
import cn.com.duibaboot.ext.autoconfigure.cloud.netflix.feign.CustomRequestInterceptor;
import com.alibaba.fastjson.JSONObject;
import com.google.common.collect.Sets;
import java.util.Optional;
import java.util.Set;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import org.jetbrains.annotations.NotNull;
import org.springframework.core.annotation.AnnotationUtils;
import org.springframework.web.method.HandlerMethod;
import org.springframework.web.servlet.handler.HandlerInterceptorAdapter;
import org.springframework.web.servlet.resource.ResourceHttpRequestHandler;

/**
 * 全局登录状态拦截器
 *
 * @author xuanhongjian
 */
public class SsoInterceptor extends HandlerInterceptorAdapter {

    private static final Set<String> EXCLUDE_PATHS = Sets.newHashSet();

    static {
        EXCLUDE_PATHS.add("/favicon.ico");
        EXCLUDE_PATHS.add("/monitor/check");
        EXCLUDE_PATHS.add("/swagger-ui.html");
        EXCLUDE_PATHS.add("/swagger-resources");
        EXCLUDE_PATHS.add("/swagger-resources/configuration/ui");
        EXCLUDE_PATHS.add("/v2/api-docs");
        EXCLUDE_PATHS.add("/error");
    }

    private SsoFilterHandlerQueue queue;

    @Override
    public boolean preHandle(@NotNull HttpServletRequest request,
        @NotNull HttpServletResponse response,
        @NotNull Object handler) throws Exception {
        //静态资源放行
        if (handler instanceof ResourceHttpRequestHandler) {
            return true;
        }
        SsoRequestTool.setRequestInThreadLocal(request, response);
        try {
            //放行rpc和特殊请求
            if (isRpc(request) || excludeSpecialRequest(request, handler)) {
                return Boolean.TRUE;
            }
            return queue.doHandler(handler);
        } catch (Exception e) {
            if (SsoRequestTool.isAsynchronousRequests()) {
                response.setHeader("Content-Type", "application/json;charset=UTF-8");
                response.getWriter().write(JSONObject.toJSONString(JsonRender.error(e.getMessage())));
            } else {
                response.setHeader("Content-Type", "text/html;charset=UTF-8");
                response.getWriter().write(e.getMessage());
            }
            return false;
        }
    }

    @Override
    public void afterCompletion(@NotNull HttpServletRequest request, @NotNull HttpServletResponse response, @NotNull Object handler, Exception ex) {
        SsoRequestTool.removeRequestInThreadLocal();
    }

    public void setHandleQueue(SsoFilterHandlerQueue queue) {
        this.queue = queue;
    }

    /**
     * 排除不需要拦截的特殊请求
     */
    private Boolean excludeSpecialRequest(@NotNull HttpServletRequest request, Object handler) {

        Boolean accessFlag = Optional.ofNullable(
            (Boolean) request.getAttribute(SsoConstants.CAN_ACCESS)).orElse(false);
        if (accessFlag) {
            return true;
        }

        String url = request.getRequestURI();
        if (EXCLUDE_PATHS.contains(url)) {
            return true;
        }

        if (handler instanceof HandlerMethod) {
            HandlerMethod method = (HandlerMethod) handler;
            CanAccess canAccess = AnnotationUtils
                                      .findAnnotation(method.getMethod(), CanAccess.class);
            if (canAccess != null) {
                return true;
            }
        }

        return Boolean.FALSE;
    }

    /**
     * 判断是否rpc
     */
    private Boolean isRpc(@NotNull HttpServletRequest request) {
        //rpc请求直接放行
        if (Boolean.TRUE.toString().equals(request.getHeader(CustomRequestInterceptor.X_RPC))) {
            return Boolean.TRUE;
        }
        return Boolean.FALSE;
    }
}
