package cn.com.duiba.cloud.single.sign.on.contract.tool;


import cn.com.duiba.cloud.single.sign.on.contract.common.exception.SsoRunTimeException;
import cn.com.duiba.wolf.perf.timeprofile.RequestTool;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import java.util.List;
import java.util.Map;
import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import org.apache.commons.lang.StringUtils;

/**
 * @author xuanhongjian
 */
public class SsoRequestTool {

    private static final ThreadLocal<SsoRequestTool> LOCAL = new ThreadLocal<>();
    private static final ThreadLocal<RequestParams> TTL = new ThreadLocal<>();
    private HttpServletRequest request;
    private HttpServletResponse response;


    private static SsoRequestTool get() {
        if (LOCAL.get() == null) {
            throw new SsoRunTimeException(
                "请在登录拦截器中调用setRequestInThreadLocal,或者请检查是否在异步线程中使用RequestTool,"
                    + "如果有异步场景要获取用户信息,请在异步线程启动之前使用getRequestParams获取当前用户的请求封装信息");
        }
        return LOCAL.get();
    }

    public static void addCookie(Cookie cookie) {
        if (cookie == null) {
            return;
        }
        SsoRequestTool tool = SsoRequestTool.get();
        if (tool == null) {
            throw new SsoRunTimeException("用户线程已经结束");
        }
        RequestParams params = getRequestParams();
        params.addCookie(cookie.getName(), cookie.getValue());
        tool.response.addCookie(cookie);
    }

    public static RequestParams getRequestParams() {
        RequestParams params = TTL.get();
        if (params == null) {
            throw new SsoRunTimeException("非用户请求线程，无法获取用户请求数据");
        }
        return params;
    }

    public static String getCookie(String name) {
        RequestParams params = getRequestParams();
        return params.getCookie(name);
    }


    public static Boolean isAsynchronousRequests() {
        HttpServletRequest request = get().request;
        String re = request.getHeader("x-requested-with");
        String accept = request.getHeader("Accept");
        return (StringUtils.isNotBlank(re) && StringUtils.equals("XMLHttpRequest", re)) || (
            StringUtils.isNotBlank(accept) && accept.contains("json"));
    }

    public static HttpServletResponse getResponse() {
        return get().response;
    }

    public static HttpServletRequest getRequest() {
        return get().request;
    }


    private static String parseHomeUrl(HttpServletRequest request) {
        StringBuilder url = new StringBuilder();

        String scheme = request.getScheme();
        int port = request.getServerPort();
        url.append(scheme);
        url.append("://");
        url.append(request.getServerName());
        List<Integer> defaultPortList = Lists.newArrayList(80,443,0);
        if (defaultPortList.contains(port)) {
            return url.toString();
        }
        url.append(':');
        url.append(request.getServerPort());
        return url.toString();
    }
    public static String getHomeUrl() {
        return parseHomeUrl(SsoRequestTool.getRequest());
    }

    public static String getUrl(){
        RequestParams params = getRequestParams();
        return params.getUrl();
    }



    public static void setRequestInThreadLocal(HttpServletRequest request, HttpServletResponse response){
        if(request==null){
            throw new SsoRunTimeException("RequestTool注入request为空");
        }
        if(response==null){
            throw new SsoRunTimeException("RequestTool注入response为空");
        }
        SsoRequestTool tool = new SsoRequestTool();
        tool.request = request;
        tool.response = response;
        TTL.set(createRequestParams(request));
        LOCAL.set(tool);
    }


    private static RequestParams createRequestParams(HttpServletRequest request){
        if(request==null) {
            throw new SsoRunTimeException("RequestParams解析时的request不能为空");
        }
        RequestParams rp = new RequestParams();

        Map<String,String> cookieMap = parseCookies(request);
        rp.setUrl(parseUrl(request));
        rp.setCookies(cookieMap);
        rp.setIp(parseIp(request));
        rp.setUri(request.getRequestURI());
        return rp;
    }

    private static Map<String,String> parseCookies(HttpServletRequest request){
        Cookie[] cookies=request.getCookies();
        Map<String,String> cookieMap = Maps.newHashMap();
        if(cookies!=null){
            for(Cookie cookie : cookies){
                cookieMap.put(cookie.getName(),cookie.getValue());
            }
        }
        return cookieMap;
    }

    private static String parseUrl(HttpServletRequest request){
        StringBuilder stringBuilder = new StringBuilder();
        stringBuilder.append(parseHomeUrl(request));
        String urlPath = request.getRequestURI();
        if(StringUtils.isNotBlank(urlPath) && !StringUtils.equals("/",urlPath)){
            stringBuilder.append(urlPath);
        }
        String queryString = request.getQueryString();
        if (StringUtils.isNotBlank(queryString)) {
            stringBuilder.append("?").append(queryString);
        }
        return stringBuilder.toString();
    }

    public static String parseIp(HttpServletRequest request) {
        return RequestTool.getIpAddr(request);
    }

    /**
     * 提取request中的参数
     *
     * @return 参数列表
     */
    public static Map<String, String> getRequestParamMap() {
        HttpServletRequest request = get().request;
        Map<String, String[]> map = request.getParameterMap();
        Map<String, String> resultMap = Maps.newHashMap();
        for (Map.Entry<String, String[]> entry : map.entrySet()) {
            String[] values = entry.getValue();
            if (values.length != 0) {
                resultMap.put(entry.getKey(), entry.getValue()[0]);
            }
        }
        return resultMap;
    }

    public static void removeRequestInThreadLocal(){
        LOCAL.remove();
        TTL.remove();
    }
}
