package cn.com.duiba.sso.api.tool;

import cn.com.duiba.sso.api.domain.dto.AdminDto;
import cn.com.duiba.sso.api.exception.SsoRunTimeException;
import com.google.common.collect.Maps;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import java.io.Serializable;
import java.util.Collections;
import java.util.Map;

/**
 * Created by liuyao on 2017/5/11.
 */
public class RequestParams implements Serializable,Cloneable {

    private transient Logger logger = LoggerFactory.getLogger(RequestParams.class);

    public static final String LOCALHOST_IP="127.0.0.1";

    private Map<String, String> cookies = Collections.emptyMap();
    private String url;
    private String uri;
    private String ip;
    private AdminDto admin;

    public String getUrl() {
        return url;
    }

    public String getUri(){
        return uri;
    }

    protected void addCookie(String name, String value){
        cookies.put(name,value);
    }

    public String getCookie(String name) {
        return this.cookies.get(name);
    }

    public AdminDto getAdmin() {
        return admin;
    }

    public void setAdmin(AdminDto admin) {
        this.admin = admin;
    }

    public String getIp(){
        return ip;
    }

    public static RequestParams parse(HttpServletRequest request) {
        if(request==null) throw new SsoRunTimeException("RequestParams解析时的request不能为空");
        RequestParams rp = new RequestParams();

        Map<String,String> cookieMap = parseCookies(request);
        rp.url = parseUrl(request);
        rp.cookies = cookieMap;
        rp.ip = parseIp(request);
        rp.uri = request.getRequestURI();
        return rp;
    }

    private static String parseUrl(HttpServletRequest request){
        String url = request.getRequestURL().toString();
        String queryString = request.getQueryString();
        StringBuilder sb = new StringBuilder();
        sb.append(url);
        if(StringUtils.isNotBlank(queryString)) sb.append("?").append(queryString);
        return sb.toString();
    }

    private static Map<String,String> parseCookies(HttpServletRequest request){
        Cookie[] cookies=request.getCookies();
        Map<String,String> cookieMap = Maps.newHashMap();
        if(cookies!=null){
            for(Cookie cookie : cookies){
                cookieMap.put(cookie.getName(),cookie.getValue());
            }
        }
        return cookieMap;
    }

    private static String parseIp(HttpServletRequest request){
        String ip = request.getHeader("x-forwarded-for");
        if (ip != null && ip.trim().length() > 0) {
            String[] ips = ip.trim().split(",");
            int size = ips.length;
            if (size > 0) {
                ip = ips[0].trim();
            }
        }
        if (ip == null || ip.length() == 0 || "unknown".equalsIgnoreCase(ip)) {
            ip = request.getHeader("X-Real-IP");
        }
        if (ip == null || ip.length() == 0 || "unknown".equalsIgnoreCase(ip)) {
            ip = request.getHeader("Proxy-Client-IP");
        }
        if (ip == null || ip.length() == 0 || "unknown".equalsIgnoreCase(ip)) {
            ip = request.getHeader("WL-Proxy-Client-IP");
        }
        if (ip == null || ip.length() == 0 || "unknown".equalsIgnoreCase(ip)) {
            ip = request.getHeader("Cdn-Src-Ip");
        }
        if (ip == null || ip.length() == 0 || "unknown".equalsIgnoreCase(ip)) {
            ip = request.getRemoteAddr();
        }
        if (ip != null && ip.startsWith("0:0:0:0")) {
            ip = LOCALHOST_IP;
        }
        return ip;
    }

    @Override
    protected RequestParams clone(){
        try {
            return (RequestParams)super.clone();
        }catch (Exception e){
            logger.error("RequestParams复制失败",e);
            throw new SsoRunTimeException(e);
        }
    }
}
