package cn.com.duiba.sso.api.web.interceptor.handler.impl;

import cn.com.duiba.sso.api.constants.SsoProperties;
import cn.com.duiba.sso.api.domain.dto.AdminDto;
import cn.com.duiba.sso.api.exception.SsoRunTimeException;
import cn.com.duiba.sso.api.remoteservice.RemoteSSOService;
import cn.com.duiba.sso.api.service.OutLoginEvent;
import cn.com.duiba.sso.api.tool.CookieUtil;
import cn.com.duiba.sso.api.tool.RequestTool;
import cn.com.duiba.sso.api.tool.SystemInfo;
import cn.com.duiba.sso.api.web.interceptor.handler.SsoFilterHandler;
import com.alibaba.fastjson.JSONObject;
import com.github.benmanes.caffeine.cache.Cache;
import com.github.benmanes.caffeine.cache.CacheLoader;
import com.github.benmanes.caffeine.cache.Caffeine;
import com.github.benmanes.caffeine.cache.LoadingCache;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.context.event.EventListener;

import javax.annotation.Nonnull;
import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import java.net.URLEncoder;
import java.util.Objects;
import java.util.concurrent.TimeUnit;

/**
 * Created by liuyao on 2017/5/11.
 * 单点登录拦截系统
 */
@Slf4j
public class LoginFilterHandler implements SsoFilterHandler {

    private LoadingCache<String, AdminDto> adminDtoCache = Caffeine.newBuilder().expireAfterWrite(5, TimeUnit.SECONDS).build(new CacheLoader<String, AdminDto>() {
        @Override
        public AdminDto load(@Nonnull String ticket) {
            AdminDto adminDao = remoteSSOService.verifyTicketAndGetAdmin(ticket);
            if(adminDao==null){
                adminDao = new AdminDto();
            }
            return adminDao;
        }
    });

    @Resource
    private SsoProperties ssoProperties;
    @Resource
    private RemoteSSOService remoteSSOService;

    @Override
    public Boolean before(Object handler) {


        String ticket = RequestTool.findTicket();
        if(StringUtils.isBlank(ticket)){
            returnNotLogin();
            return false;
        }

        AdminDto admin = verifyTicket(ticket);
        if(admin==null){
            returnNotLogin();
            return false;
        }
        //注入adminId
        RequestTool.setAdmin(admin);
        return true;
    }

    @EventListener(OutLoginEvent.class)
    public void outLoginEventListener(OutLoginEvent event){
        adminDtoCache.invalidate(event.getTicket());
    }

    /**
     * 获取登录重定向地址
     */
    private String getLoginRedirectUrl(){
        String redirect = getCurrentRedirect();

        String ssoHomeUrl = remoteSSOService.findSsoHomeUrl();

        String loginUrl = ssoHomeUrl + "/login";
        return loginUrl + "?redirect=" + redirect + "&systemId=" + SystemInfo.getThisSystemId();
    }

    private String getDefaultRedirectUrl(){
        if(RequestTool.isMobile()){
            return ssoProperties.getMobileDefaultRedirectUrl();
        }else {
            return ssoProperties.getDefaultRedirectUrl();
        }
    }

    @Override
    public int getOrder() {
        return 0;
    }

    /**
     * 验证通行证
     */
    protected AdminDto verifyTicket(String ticket){
        if(StringUtils.isBlank(ticket)){
            return null;
        }
        AdminDto admin = adminDtoCache.get(ticket);
        if(Objects.requireNonNull(admin).getId()==null){
            return null;
        }
        //有可能是根据链接传入的通行证，把它持久化到客户端的cookie中
        if(StringUtils.isBlank(RequestTool.getCookie(CookieUtil.LOGIN_COOKIE_NAME))){
            CookieUtil.setLoginCookie(ticket);
        }
        return admin;
    }

    /**
     * 把当前请求包装成重定向参数
     */
    protected String getCurrentRedirect(){
        HttpServletRequest request = RequestTool.getRequest();
        String redirect;
        if(StringUtils.equals("/",request.getRequestURI())){
            redirect = RequestTool.getHomeURL()+getDefaultRedirectUrl();
        }else{
            redirect = RequestTool.getUrl();
        }
        try{
            return URLEncoder.encode(redirect,"utf-8");
        }catch (Exception e){
            throw new SsoRunTimeException(e);
        }
    }

    protected void returnNotLogin(){
        try{
            if(RequestTool.isAsynchronousRequests()){
                RequestTool.getResponse().setHeader("Content-Type", "application/json;charset=UTF-8");
                JSONObject model = new JSONObject();
                model.put("code","SSO:01001");
                model.put("success",false);
                model.put("notLogin",true);
                model.put("message","登录失效，请刷新页面");
                RequestTool.getResponse().getWriter().write(model.toJSONString());
            }else{
                //附上当前请求地址,重定向到SSO登录页
                RequestTool.getResponse().sendRedirect(getLoginRedirectUrl());
            }
        }catch (Exception e){
            throw new SsoRunTimeException(e);
        }
    }

}
