package cn.com.duiba.sso.api.web.interceptor.handler.impl;

import cn.com.duiba.sso.api.constants.SsoProperties;
import cn.com.duiba.sso.api.domain.dto.LoginStateDto;
import cn.com.duiba.sso.api.exception.SsoRunTimeException;
import cn.com.duiba.sso.api.remoteservice.RemoteSSOService;
import cn.com.duiba.sso.api.service.OutLoginEvent;
import cn.com.duiba.sso.api.tool.CookieUtil;
import cn.com.duiba.sso.api.tool.RequestTool;
import cn.com.duiba.sso.api.tool.SystemInfo;
import cn.com.duiba.sso.api.web.interceptor.handler.SsoFilterHandler;
import cn.com.duiba.wolf.utils.UrlUtils2;
import com.alibaba.fastjson.JSONObject;
import com.github.benmanes.caffeine.cache.CacheLoader;
import com.github.benmanes.caffeine.cache.Caffeine;
import com.github.benmanes.caffeine.cache.LoadingCache;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.context.event.EventListener;

import javax.annotation.Nonnull;
import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import java.net.URLEncoder;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.TimeUnit;

/**
 *
 * @author liuyao
 * @date 2017/5/11
 * 单点登录拦截系统
 */
@Slf4j
public class LoginFilterHandler implements SsoFilterHandler {

    @Resource
    private SsoProperties ssoProperties;
    @Resource
    private RemoteSSOService remoteSSOService;

    private final LoadingCache<String, LoginStateDto> loginStateCache = Caffeine.newBuilder().expireAfterWrite(5, TimeUnit.SECONDS).build(new CacheLoader<String, LoginStateDto>() {
        @Override
        public LoginStateDto load(@Nonnull String ticket) {
            return remoteSSOService.verifyTicket(ticket);
        }
    });

    @EventListener(OutLoginEvent.class)
    public void outLoginEventListener(OutLoginEvent event){
        loginStateCache.invalidate(event.getTicket());
    }

    @Override
    public Boolean before(Object handler) {
        String ticket = RequestTool.findTicket();
        if(StringUtils.isBlank(ticket)){
            returnNotLogin();
            return false;
        }
        LoginStateDto loginState = verifyTicket(ticket);
        if(loginState==null){
            returnNotLogin();
            return false;
        }
        //注入adminId
        RequestTool.setLoginState(loginState);
        return true;
    }

    /**
     * 获取登录重定向地址
     */
    private String getLoginRedirectUrl(){
        Map<String,String> params = Maps.newHashMap();
        String redirect = getCurrentRedirect();
        params.put("redirect",redirect);
        params.put("systemId",SystemInfo.getThisSystemId().toString());
        String ssoHomeUrl = remoteSSOService.findSsoHomeUrl();
        String loginUrl = ssoHomeUrl + "/login";
        return UrlUtils2.appendParams(loginUrl,params);
    }

    private String getDefaultRedirectUrl(){
        if(RequestTool.isMobile()){
            return ssoProperties.getMobileDefaultRedirectUrl();
        }else {
            return ssoProperties.getDefaultRedirectUrl();
        }
    }

    @Override
    public int getOrder() {
        return 0;
    }

    /**
     * 验证通行证
     */
    protected LoginStateDto verifyTicket(String ticket){
        if(StringUtils.isBlank(ticket)){
            return null;
        }
        LoginStateDto loginState = loginStateCache.get(ticket);
        if(Objects.isNull(loginState)){
            return null;
        }
        //有可能是根据链接传入的通行证，把它持久化到客户端的cookie中
        if(StringUtils.isBlank(RequestTool.getCookie(CookieUtil.LOGIN_COOKIE_NAME))){
            CookieUtil.setLoginCookie(ticket);
        }
        return loginState;
    }

    /**
     * 把当前请求包装成重定向参数
     */
    protected String getCurrentRedirect(){
        HttpServletRequest request = RequestTool.getRequest();
        String redirect;
        if(StringUtils.equals("/",request.getRequestURI())){
            redirect = RequestTool.getHomeURL()+getDefaultRedirectUrl();
        }else{
            redirect = RequestTool.getUrl();
        }
        try{
            return URLEncoder.encode(redirect,"utf-8");
        }catch (Exception e){
            throw new SsoRunTimeException(e);
        }
    }

    protected void returnNotLogin(){
        try{
            if(RequestTool.isAsynchronousRequests()){
                RequestTool.getResponse().setHeader("Content-Type", "application/json;charset=UTF-8");
                JSONObject model = new JSONObject();
                model.put("code","SSO:01001");
                model.put("success",false);
                model.put("notLogin",true);
                model.put("message","登录失效，请刷新页面");
                RequestTool.getResponse().getWriter().write(model.toJSONString());
            }else{
                //附上当前请求地址,重定向到SSO登录页
                RequestTool.getResponse().sendRedirect(getLoginRedirectUrl());
            }
        }catch (Exception e){
            throw new SsoRunTimeException(e);
        }
    }

}
