package cn.com.duiba.sso.api.tool;

import cn.com.duiba.sso.api.domain.dto.AdminDto;
import cn.com.duiba.sso.api.domain.dto.LoginStateDto;
import cn.com.duiba.sso.api.exception.SsoException;
import cn.com.duiba.sso.api.exception.SsoRunTimeException;
import cn.com.duiba.sso.api.remoteservice.RemoteAdminService;
import cn.com.duiba.sso.api.remoteservice.RemoteSSOService;
import cn.com.duiba.sso.api.service.OutLoginEvent;
import cn.com.duiba.sso.api.web.interceptor.MobileInterceptor;
import cn.com.duiba.wolf.utils.UUIDUtils;
import com.github.benmanes.caffeine.cache.CacheLoader;
import com.github.benmanes.caffeine.cache.Caffeine;
import com.github.benmanes.caffeine.cache.LoadingCache;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.checkerframework.checker.nullness.qual.NonNull;
import org.checkerframework.checker.nullness.qual.Nullable;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;

import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.TimeUnit;

/**
 * Created by liuyao on 2017/5/11.
 */
@Slf4j
public class RequestTool{

    private static final ThreadLocal<RequestTool> local = new ThreadLocal<>();
    private static final ThreadLocal<RequestParams> TTL = new ThreadLocal<>();

    private static RemoteAdminService remoteAdminService;
    private static RemoteSSOService remoteSSOService;
    private static ApplicationContext applicationContext;

    private static final LoadingCache<Long,AdminDto> adminCache = Caffeine.newBuilder().expireAfterWrite(1, TimeUnit.MINUTES).build(new CacheLoader<Long, AdminDto>() {
        @Nullable
        @Override
        public AdminDto load(@NonNull Long adminId) {
            try{
                return remoteAdminService.findOneAdmin(adminId);
            }catch (Exception e){
                log.error("拉取用户信息失败",e);
                throw new IllegalArgumentException(e);
            }
        }
    });

    private HttpServletRequest request;
    private HttpServletResponse response;

    private final String uuid = UUIDUtils.createUUID();

    public static void setRequestInThreadLocal(HttpServletRequest request, HttpServletResponse response){
        if(request==null){
            throw new SsoRunTimeException("RequestTool注入request为空");
        }
        if(response==null){
            throw new SsoRunTimeException("RequestTool注入response为空");
        }
        RequestTool tool = new RequestTool();
        tool.request = request;
        tool.response = response;
        TTL.set(createRequestParams(request));
        local.set(tool);
    }

    private static RequestTool get(){
        if(local.get() == null){
            throw new SsoRunTimeException("请在登录拦截器中调用setRequestInThreadLocal,或者请检查是否在异步线程中使用RequestTool,如果有异步场景要获取用户信息,请在异步线程启动之前使用getRequestParams获取当前用户的请求封装信息");
        }
        return local.get();
    }

    public static RequestParams getRequestParams(){
        RequestParams params = TTL.get();
        if(params==null){
            throw new SsoRunTimeException("非用户请求线程，无法获取用户请求数据");
        }
        return params;
    }

    public static HttpServletResponse getResponse(){
        return get().response;
    }

    public static HttpServletRequest getRequest(){
        return get().request;
    }

    public static void removeRequestInThreadLocal(){
        local.remove();
        TTL.remove();
    }

    /**
     * 获取管理员Id
     * @return adminId
     */
    public static Long getAdminId(){
        RequestParams param = getRequestParams();
        LoginStateDto loginState = param.getLoginState();
        return Optional.ofNullable(loginState).map(LoginStateDto::getAdminId).orElse(null);
    }

    public static AdminDto getAdmin() {
        Long adminId = getAdminId();
        if(Objects.isNull(adminId)){
            throw new IllegalArgumentException("非登录状态，无法获取");
        }
        return adminCache.get(adminId);
    }

    public static void setLoginState(LoginStateDto loginState) {
        RequestParams param = getRequestParams();
        param.setLoginState(loginState);
    }

    public static void addCookie(Cookie cookie){
        if(cookie==null){
            return;
        }
        RequestTool tool = RequestTool.get();
        if(tool==null){
            throw new SsoRunTimeException("用户线程已经结束");
        }
        RequestParams params = getRequestParams();
        params.addCookie(cookie.getName(),cookie.getValue());
        tool.response.addCookie(cookie);
    }

    public static String getUrl(){
        RequestParams params = getRequestParams();
        return params.getUrl();
    }

    public static boolean isHttpsRequest(){
        HttpServletRequest request= get().request;
        return Boolean.parseBoolean(request.getHeader("Use-Https"));
    }

    public static String getCookie(String name){
        RequestParams params = getRequestParams();
        return params.getCookie(name);
    }

    /**
     * 提取request中的参数
     * @return 参数列表
     */
    public static Map<String,String> getRequestParamMap(){
        HttpServletRequest request= get().request;
        Map<String, String[]> map=request.getParameterMap();
        Map<String, String> resultMap=new HashMap<>();
        for(Map.Entry<String, String[]> entry :map.entrySet()){
            String[] values = entry.getValue();
            if(values.length!=0){
                resultMap.put(entry.getKey(), entry.getValue()[0]);
            }
        }
        return resultMap;
    }

    public static Boolean isAsynchronousRequests(){
        HttpServletRequest request= get().request;
        String re = request.getHeader("x-requested-with");
        String accept = request.getHeader("Accept");
        return (StringUtils.isNotBlank(re) && StringUtils.equals("XMLHttpRequest",re)) || (StringUtils.isNotBlank(accept) && accept.contains("json"));
    }

    public static String getIp(){
        RequestParams params = getRequestParams();
        return params.getIp();
    }

    public static String getRequestUUID(){
        RequestTool tool = get();
        return tool.uuid;
    }

    public static Boolean isLocalHost(){
        RequestParams params = getRequestParams();
        return Objects.equals(params.getIp(), "127.0.0.1");//NOSONAR
    }

    public static void googleVerify(String code) throws SsoException{
        Long adminId = getAdminId();
        Boolean result = remoteAdminService.googleCodeVerify(adminId,code);
        if(!result){
            throw new SsoException("动态码验证失败");
        }
    }

    public static String getHomeURL(){
        return parseHomeUrl(RequestTool.getRequest());
    }

    /**
     * 获取令牌
     * @return SSO通行证
     */
    public static String findTicket(){
        String ticketForUrl = getRequestParamMap().get(CookieUtil.LOGIN_COOKIE_NAME);
        if(StringUtils.isNotBlank(ticketForUrl)){
            return ticketForUrl;
        }
        return CookieUtil.getCookie(CookieUtil.LOGIN_COOKIE_NAME);
    }

    public static String findDeviceNo(){
        String deviceNo = CookieUtil.getCookie(CookieUtil.DEVICE_NO);
        if(StringUtils.isBlank(deviceNo)){
            deviceNo = remoteSSOService.findDeviceNoByTicket(findTicket());
        }
        return deviceNo;
    }

    public static void outLogin(){
        String ticket = RequestTool.findTicket();
        if(StringUtils.isBlank(ticket)){
            return;
        }
        remoteSSOService.outLogin(ticket);
        OutLoginEvent event = new OutLoginEvent();
        event.setTicket(ticket);
        applicationContext.publishEvent(event);

    }

    public static Boolean isMobile(){
        HttpServletRequest request= get().request;
        return Optional.ofNullable((Boolean)request.getAttribute(MobileInterceptor.IS_MOBILE_MARK)).orElse(false);
    }

    private static RequestParams createRequestParams(HttpServletRequest request){
        if(request==null) {
            throw new SsoRunTimeException("RequestParams解析时的request不能为空");
        }
        RequestParams rp = new RequestParams();

        Map<String,String> cookieMap = parseCookies(request);
        rp.setUrl(parseUrl(request));
        rp.setCookies(cookieMap);
        rp.setIp(parseIp(request));
        rp.setUri(request.getRequestURI());
        return rp;
    }

    @Autowired
    public void setRemoteAdminService(RemoteAdminService remoteAdminService) {
        RequestTool.remoteAdminService = remoteAdminService;//NOSONAR
    }

    @Autowired
    public void setRemoteSSOService(RemoteSSOService remoteSSOService) {
        RequestTool.remoteSSOService = remoteSSOService;//NOSONAR
    }

    @Autowired
    public void setApplicationContext(ApplicationContext applicationContext) {
        RequestTool.applicationContext = applicationContext;//NOSONAR
    }

    private static String parseUrl(HttpServletRequest request){
        StringBuilder stringBuilder = new StringBuilder();
        stringBuilder.append(parseHomeUrl(request));
        String urlPath = request.getRequestURI();
        if(StringUtils.isNotBlank(urlPath) && !StringUtils.equals("/",urlPath)){
            stringBuilder.append(urlPath);
        }
        String queryString = request.getQueryString();
        if(StringUtils.isNotBlank(queryString)) {
            stringBuilder.append("?").append(queryString);
        }
        return stringBuilder.toString();
    }

    private static String parseHomeUrl(HttpServletRequest request){
        StringBuilder url = new StringBuilder();

        String scheme = request.getScheme ();
        int port = request.getServerPort ();
        url.append (scheme);
        url.append ("://");
        url.append (request.getServerName ());

        if(port==80 || port==443 || port<=0){
            return url.toString();
        }
        url.append (':');
        url.append (request.getServerPort());
        return url.toString();
    }

    private static Map<String,String> parseCookies(HttpServletRequest request){
        Cookie[] cookies=request.getCookies();
        Map<String,String> cookieMap = Maps.newHashMap();
        if(cookies!=null){
            for(Cookie cookie : cookies){
                cookieMap.put(cookie.getName(),cookie.getValue());
            }
        }
        return cookieMap;
    }

    public static String parseIp(HttpServletRequest request){
        return cn.com.duiba.wolf.perf.timeprofile.RequestTool.getIpAddr(request);
    }

}
