package cn.com.duiba.sso.api.tool;

import cn.com.duiba.boot.event.MainContextRefreshedEvent;
import cn.com.duiba.sso.api.domain.dto.SystemDto;
import cn.com.duiba.sso.api.domain.event.SsoSystemInitSuccessEvent;
import cn.com.duiba.sso.api.domain.properties.SpringApplicationProperties;
import cn.com.duiba.sso.api.exception.SsoRunTimeException;
import cn.com.duiba.sso.api.remoteservice.RemoteSystemService;
import cn.com.duiba.sso.api.service.eventbus.SsoEventBus;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.event.EventListener;
import org.springframework.stereotype.Component;

import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.atomic.AtomicReference;

/**
 * Created by liuyao on 2017/12/5.
 * 本系统的Mate信息
 */
@Component
public class SystemInfo {

    private Logger logger = LoggerFactory.getLogger(SystemInfo.class);

    private static AtomicReference<SystemDto> system =new AtomicReference<>();

    private static RemoteSystemService remoteSystemService;

    private static volatile String systemAlias;

    private static Long NO_REGIST_SYS_ID = -1L;

    @Autowired
    private SsoEventBus eventBus;
    @Autowired
    private ExecutorService executorService;

    @Autowired
    public void setRemoteSystemService(RemoteSystemService remoteSystemService) {
        SystemInfo.remoteSystemService = remoteSystemService;
    }

    @Autowired
    private SpringApplicationProperties springApplicationProperties;


    @EventListener(MainContextRefreshedEvent.class)
    public void MainContextRefreshedEventListener(){

        SystemInfo.systemAlias = Optional.ofNullable(springApplicationProperties.getPriorName()).orElse(springApplicationProperties.getName());
        if (StringUtils.isBlank(SystemInfo.systemAlias)){
            throw new SsoRunTimeException("Sso客户端启动失败，请配置 spring.application.name 系统别名");
        }

        executorService.submit(new Runnable() {

            private Integer count = 10;//最多加载重试10

            @Override
            public void run() {
                do{
                    try{
                        SystemDto systemDto = remoteSystemService.getSystemByAlias(SystemInfo.systemAlias);
                        if(systemDto==null){
                            systemDto = new SystemDto();
                            systemDto.setId(NO_REGIST_SYS_ID);
                        }
                        system.set(systemDto);
                        String version = SsoVersion.getVersion();
                        if(StringUtils.isNotBlank(version)){
                            remoteSystemService.submitSsoVersion(systemDto.getId(),version);
                        }
                        break;
                    }catch (Exception e){
                        logger.error("加载系统信息失败",e);
                    }finally {
                        this.count--;
                    }
                }while (count>0 && system.get()==null);

                if(!Objects.equals(NO_REGIST_SYS_ID,system.get().getId())){
                    eventBus.post(new SsoSystemInitSuccessEvent());
                }
            }
        });
    }

    /**
     * 获取本系统的信息
     * @return
     */
    public static SystemDto getThisSystemInfo(){
        while(true){
            //自旋等待
            if (!(system.get() == null)) break;
        }
        SystemDto systemDto = system.get();
        if(Objects.equals(NO_REGIST_SYS_ID,systemDto.getId())){
            throw new SsoRunTimeException("未找到"+systemAlias+"对应的系统配置");
        }
        return systemDto;
    }


    /**
     * 获取系统别名
     * @return
     */
    public static String getThisSystemAlias(){
        return systemAlias;
    }



    /**
     * 获取本系统的Id
     * @return
     */
    public static Long getThisSystemId(){
        return getThisSystemInfo().getId();
    }
}
