package cn.com.duiba.sso.api.service.export;

import cn.com.duiba.sso.api.exception.SsoRunTimeException;
import com.google.common.base.Joiner;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Scope;
import org.springframework.data.redis.core.BoundHashOperations;
import org.springframework.data.redis.core.StringRedisTemplate;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import java.io.File;
import java.util.Date;
import java.util.Objects;

@Scope("prototype")
public abstract class BaseExportRunnable implements Runnable {

    private Logger logger = LoggerFactory.getLogger(BaseExportRunnable.class);

    protected static final Joiner FILE_JOINER = Joiner.on(",").useForNull("");

    private final Long taskId = new Date().getTime();

    private File file;

    @Autowired
    private SsoExportFileService ssoExportFileService;

    private BoundHashOperations<String,String,String> ops;

    @PostConstruct
    public void init(){

        createFile();
        String key = ssoExportFileService.getTaskKey(taskId);
        StringRedisTemplate stringRedisTemplate = ssoExportFileService.getStringRedisTemplate();
        if(stringRedisTemplate==null){
            throw new SsoRunTimeException("SSO任务依赖StringRedisTemplate");
        }

        ops = stringRedisTemplate.boundHashOps(key);

        ops.put("successCount", "0");
        ops.put("count", "0");
        ops.put("success", "false");
        ops.put("url", "");
        ops.put("fileName", file.getName());
        ops.put("taskId", taskId.toString());
    }

    @PreDestroy
    public void des(){
        if(file!=null&&file.exists()&&file.delete()){
            logger.debug("执行文件清理");
        }
    }

    /**
     * 获取导出任务Id
     * @return
     */
    public Long getTaskId(){
        return taskId;
    }

    protected File getTempFile(){
        return file;
    }

    protected void setIncrementCount(Integer count){
        Objects.requireNonNull(count);
        ops.increment("successCount",count);
    }

    protected void setCount(Integer count){
        ops.put("count", count.toString());
    }

    protected void setMessage(String message){
        ops.put("message", message);
    }

    /**
     * 导出完成之后，下发给客户端一个能够全网访问的文件地址
     * @param url
     */
    protected void complete(String url){
        ops.put("url", url);
        ops.put("success", "true");
    }

    protected void error(String message){
        ops.put("message", message);
        ops.put("success", "true");
    }



    private void createFile(){
        File tempDir = ssoExportFileService.getTempDir();
        file =  new File(tempDir, String.valueOf(new Date().getTime()) + ".csv");
    }

    protected String useForNull(Object obj) {
        StringBuilder builder = new StringBuilder();
        builder.append("\"");
        if (Objects.equals(null, obj)) {
            builder.append("");
        } else {
            builder.append(obj.toString());
        }
        builder.append("\"");
        return builder.toString();
    }

    protected abstract String getOssPath();


}
