package cn.com.duiba.sso.api.service.power;

import cn.com.duiba.sso.api.common.tree.Tree;
import cn.com.duiba.sso.api.common.tree.TreeFactory;
import cn.com.duiba.sso.api.common.tree.TreeView;
import cn.com.duiba.sso.api.domain.dto.PowerDto;
import cn.com.duiba.sso.api.exception.SsoRunTimeException;
import com.google.common.collect.HashMultimap;
import com.google.common.collect.Lists;
import lombok.Data;

import java.util.Date;
import java.util.List;
import java.util.Set;
import java.util.function.Function;

@Data
public class PowerTreeCache {

    /**
     * 权限树
     */
    private Tree<PowerDto> powerTree;
    /**
     * 权限树版本号
     */
    private Long version = new Date().getTime();
    /**
     * URL映射Map,与权限树同时更新
     */
    private HashMultimap<String,Long> urlMap;

    PowerTreeCache(Tree<PowerDto> powerTree, HashMultimap<String, Long> urlMap) {
        this.powerTree = powerTree;
        this.urlMap = urlMap;
    }

    PowerTreeCache(){
        powerTree = new Tree<>();
    }

    /**
     * 获取url命中的权限集合
     * @param url 请求链接
     * @return 命中的权限集合
     */
    public Set<Long> getPowerIdsByUrl(String url){
        return urlMap.get(url);
    }

    /**
     * 获取单个节点
     */
    public PowerDto getPower(Long powerId){
        return powerTree.getNode(powerId);
    }

    public void setPowerTree(Tree<PowerDto> powerTree) {
        this.powerTree = powerTree;
    }

    public void setVersion(Long version) {
        this.version = version;
    }

    /**
     * 获取用户的权限树
     * @param transform
     * @param <T>
     * @return
     */
    public <T extends TreeView<T>> List<T> getPowerTree(Function<PowerDto,T> transform){
        List<PowerDto> nodeList = powerTree.getTreeView();
        List<T> list = Lists.newArrayList();
        for(PowerDto power:nodeList){
            T node = transform.apply(power);
            if (node.getId()==null || node.getParentId()==null){
                throw new SsoRunTimeException("Function转化方法必须完成Id和parentId的转化");
            }
            list.add(node);
        }
        Tree<T> tree = TreeFactory.createTree(list);
        return TreeFactory.transformTree(tree);
    }

    /**
     * 获取权限在权限树上的遍历路径
     */
    public List<PowerDto> getPowerWay(Long powerId){
        return powerTree.getOneWayView(powerId);
    }

    public Long getVersion() {
        return version;
    }
}
