package cn.com.duiba.sso.api.common.tree;

import cn.com.duiba.sso.api.exception.SsoException;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;

import java.io.Serializable;
import java.util.List;
import java.util.Map;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import java.util.stream.Collectors;

public class Tree<T extends TreeNodeShim> implements Serializable {

    //树形视图,根节点
    private TreeNode rootNode;

    private Map<Long,T> objMap = Maps.newHashMap();

    private transient ReadWriteLock readWriteLock = new ReentrantReadWriteLock();

    public Tree(){
        rootNode = new TreeNode(0L,TreeNode.VIRTUAL_PARENT_ID);
    }

    public Tree(Long rootId){
        rootNode = new TreeNode(rootId,TreeNode.VIRTUAL_PARENT_ID);
    }

    public boolean containsId(Long id){
        return rootNode.containsId(id);
    }

    public int size(){
        return rootNode.getChildSize();
    }

    public void addNode(T shim){
        try{
            readWriteLock.writeLock().lock();
            TreeNode node = new TreeNode(shim);
            rootNode.addChildNode(node);
            shim.setLevel(node.getLevel());
            objMap.put(shim.getId(),shim);
        }finally{
            readWriteLock.writeLock().unlock();
        }
    }
    /**
     * 删除子节点，如果删除的节点存在子节点，则抛出异常
     * 如果树中已经导入了实体类，则在删除节点的同时，删除节点对应的实体
     * @param nodeId 要删除的子节点Id
     * @throws SsoException 存在子节点，无法删除
     */
    public void deleteNode(Long nodeId) throws Exception{
        try{
            readWriteLock.writeLock().lock();
            if(rootNode.containsId(nodeId)){
                rootNode.deleteChildNode(nodeId);
                objMap.remove(nodeId);
            }
        }catch(Exception e){
            throw new SsoException(e);
        }finally{
            readWriteLock.writeLock().unlock();
        }
    }

    /**
     * 获取完整的树形视图
     * @return 树形视图
     */
    public List<T> getTreeView(){
        try{
            readWriteLock.readLock().lock();
            List<TreeNode> list = rootNode.getAllNode();
            return list.stream().map((node) -> objMap.get(node.getId())).collect(Collectors.toList());
        }finally{
            readWriteLock.readLock().unlock();
        }
    }

    /**
     * 获取单节点
     * @param nodeId 查询节点的Id
     * @return 对应的节点
     */
    public T getNode(Long nodeId){
        return objMap.get(nodeId);
    }


    /**
     * 获取根节点到目的节点视图
     * @param destNodeId 目的节点的Id
     * @return 获取子节点到根节点的一条链路
     */
    public List<T> getOneWayView(Long destNodeId){
        List<Long> idSequence = Lists.newArrayList();
        rootNode.findNodeWay(destNodeId,idSequence);
        return idSequence.stream().map((id) -> objMap.get(id)).collect(Collectors.toList());
    }

}
