package cn.com.duiba.sso.api.service.power;

import cn.com.duiba.sso.api.domain.dto.AdminDto;
import cn.com.duiba.sso.api.domain.dto.PowerDto;
import cn.com.duiba.sso.api.remoteservice.RemoteAdminService;
import cn.com.duiba.sso.api.remoteservice.RemotePermissionService;
import cn.com.duiba.sso.api.tool.SystemInfo;
import com.github.benmanes.caffeine.cache.CacheLoader;
import com.github.benmanes.caffeine.cache.Caffeine;
import com.github.benmanes.caffeine.cache.LoadingCache;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.scheduling.annotation.Scheduled;

import java.util.Collections;
import java.util.List;
import java.util.Set;
import java.util.concurrent.TimeUnit;

/**
 * 用户拥有的权限缓存
 * @author liuyao
 */
@Slf4j
public class AdminPowerCacheService {
    /**
     * 用户级别权限配置缓存
     */
    private LoadingCache<Long,PowerCache> adminPowerCache = Caffeine.newBuilder().expireAfterWrite(30,TimeUnit.MINUTES).build(new AdminRoleCacheLoader());
    @Autowired
    private RemotePermissionService remotePermissionService;
    @Autowired
    private PowerTreeService powerTreeService;
    @Autowired
    private RemoteAdminService remoteAdminService;

    /**
     * 查询拥有某个权限的所有有效角色
     **/
    public List<AdminDto> getHasPowerAdmins(String powerPath){

        PowerTreeCache cache = powerTreeService.getPowerTreeCache(SystemInfo.getThisSystemId());

        Set<Long> pawerIds =  cache.getPowerIdsByUrl(powerPath);
        if(pawerIds.isEmpty()){
            return Collections.emptyList();
        }
        return remoteAdminService.findAdminsByPowerIds(SystemInfo.getThisSystemId(), Lists.newArrayList(pawerIds));
    }

    PowerCache getPowerCache(Long adminId){
        return adminPowerCache.get(adminId);
    }

    /**
     * 是否含有某项权限
     */
    public Boolean hasPower(Long adminId,String powerPath){
        PowerTreeCache powerTreeCache = powerTreeService.getPowerTreeCache(SystemInfo.getThisSystemId());
        Set<Long> pawerIds =  powerTreeCache.getPowerIdsByUrl(powerPath);
        if(pawerIds.isEmpty()){
            return true;
        }
        PowerCache cache = getPowerCache(adminId);
        return !Sets.intersection(pawerIds,cache.getPowerIdSet()).isEmpty();
    }

    /**
     * 获取员工拥有某项权限的Ids
     */
    public Set<Long> hasPowerIds(Long adminId,String powerPath){
        PowerTreeCache powerTreeCache = powerTreeService.getPowerTreeCache(SystemInfo.getThisSystemId());
        Set<Long> pawerIds =  powerTreeCache.getPowerIdsByUrl(powerPath);
        if(pawerIds.isEmpty()){
            return Collections.emptySet();
        }
        PowerCache cache = getPowerCache(adminId);
        return Sets.intersection(pawerIds,cache.getPowerIdSet());
    }

    /**
     * 获取用户资源白名单
     */
    public Set<String> getAllPowerRes(Long adminId){
        PowerCache cache = getPowerCache(adminId);
        Set<Long> powerIds = cache.getPowerIdSet();
        Set<String> urlSet = Sets.newHashSet();

        PowerTreeCache powerTreeCache = powerTreeService.getPowerTreeCache(SystemInfo.getThisSystemId());

        for(Long id:powerIds){
            PowerDto power = powerTreeCache.getPower(id);
            if (power==null){
                continue;
            }
            urlSet.addAll(power.getUrls());
        }
        return urlSet;
    }

    /**
     * 获取当前用户的权限版本号
     */
    public Long version(Long adminId){
        PowerCache cache = getPowerCache(adminId);
        return cache.getVersion();
    }

    /**
     * 清除缓存
     */
    public void clean(Long adminId){
        adminPowerCache.invalidate(adminId);
    }

    private class AdminRoleCacheLoader implements CacheLoader<Long, PowerCache> {

        @Override
        public PowerCache load(Long adminId){
            Long systemId = SystemInfo.getThisSystemId();
            Set<Long> powerIds =  remotePermissionService.getPowerIdsBySystemIdAndAdminId(systemId,adminId);
            PowerCache cache = new PowerCache(remotePermissionService.getAdminPowerVersion(systemId,adminId));
            cache.setPowerIdSet(powerIds);
            return cache;
        }
    }

    @Scheduled(cron = "0/15 0/1 * * * ?")
    public void adminPowerFlushTask() {
        try {
            Long systemId = SystemInfo.getThisSystemId();
            Set<Long> adminIds = adminPowerCache.asMap().keySet();
            for(Long adminId:adminIds){
                PowerCache cache = adminPowerCache.getIfPresent(adminId);
                if(cache==null){
                    continue;
                }
                Long version = remotePermissionService.getAdminPowerVersion(systemId,adminId);
                if(cache.getVersion()>=version){//如果本地的版本大于远程版本，不执行
                    continue;
                }
                adminPowerCache.invalidate(adminId);
            }
        }catch (Exception e){
            log.error("管理员权限检测失败",e);
        }
    }

}
