package cn.com.duiba.sso.ui;

import cn.com.duiba.sso.api.domain.dto.PowerDto;
import cn.com.duiba.sso.api.domain.dto.SystemDto;
import cn.com.duiba.sso.api.domain.enums.ResourceType;
import cn.com.duiba.sso.api.exception.SsoRunTimeException;
import cn.com.duiba.sso.api.service.power.PowerTreeCache;
import cn.com.duiba.sso.api.service.power.PowerTreeService;
import cn.com.duiba.sso.api.service.system.SsoSystemService;
import cn.com.duiba.sso.api.tool.JsonRender;
import cn.com.duiba.sso.api.web.interceptor.annotation.CanAccess;
import com.alibaba.fastjson.JSONObject;
import com.google.common.base.Joiner;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.io.Resource;
import org.springframework.http.CacheControl;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.web.servlet.ModelAndView;

import java.io.IOException;
import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.stream.Collectors;

import static org.springframework.web.bind.annotation.RequestMethod.GET;

@RestController
@RequestMapping("/sso/ui")
public class SsoUIController {

    private static final String UI_SYSTEM_NAME = "zt-ui-boot";

    @Autowired
    private PowerTreeService powerTreeService;
    @Autowired
    private SsoSystemService ssoSystemService;






    /**
     * 获取ui总菜单
     * 菜单项的配置是在 SSO系统 的权限——》资源管理 - 》中台UI基础项目
     *
     * 配置文档
     *
     * 资源名称：对应菜单名称
     * 资源类型：JSON
     * 资源内容：{
     *  "system":"sso-service",//归属的系统，会根据当前用户的系统权限进行校验
     *  "icon": "glyphicon-random",//一级菜单的图标，二级菜单不用配置
     *  "state": "app",//(必填)angular路由态 如果system 没有配置，该配置不会生效，强行根据url进行跳转
     *  "params":{id:1},//angular路由态的参数
     *  "url":"http://sso.duiba.com.cn/#!/system/systemList",(必填)菜单跳转链接,不需要指定域名，SSO会根据系统配置域名自动填充
     *  "keys":"app"//菜单权限校验的key  system没有指定无效
     * }
     * @return UI的菜单合集
     */
    @CanAccess
    @RequestMapping("/uiMume")
    public JsonRender uiMume(){
        SystemDto system = ssoSystemService.getSystemByAlias(UI_SYSTEM_NAME);
        PowerTreeCache cache = powerTreeService.getPowerTreeCache(system.getId());

        try{
            SsoUIMumeTransform transform = new SsoUIMumeTransform();
            transform.setCache(cache);

            List<SsoUIMume> mumes = cache.getPowerTree(transform);
            JsonRender render = JsonRender.successResult();
            render.put("mumes",mumes);
            return render;
        }catch (Exception e){
            return JsonRender.failResult(e);
        }
    }


    private class SsoUIMumeTransform implements Function<PowerDto,SsoUIMume> {

        private PowerTreeCache cache;

        private Joiner joiner = Joiner.on(".").skipNulls();

        @Override
        public SsoUIMume apply(PowerDto powerDto) {
            if(StringUtils.isBlank(powerDto.getJson())){
                throw new SsoRunTimeException("菜单项："+powerDto.getName()+"配置异常");
            }

            SsoUIMume mume = new SsoUIMume();
            mume.setId(powerDto.getId());
            mume.setParentId(powerDto.getParentId());
            mume.setName(powerDto.getName());

            JSONObject json = JSONObject.parseObject(powerDto.getJson());
            List<PowerDto> powerLinks = cache.getPowerWay(powerDto.getId());

            List<String> stateList = powerLinks.stream().map(power->{
                JSONObject item = JSONObject.parseObject(power.getJson());
                return item.getString("state");
            }).collect(Collectors.toList());

            mume.setUrl(json.getString("url"));
            mume.setState(joiner.join(stateList));
            if(json.containsKey("params")){
                mume.setParams(json.getJSONObject("params"));
            }
            if(json.containsKey("icon")){
                mume.setIcon(json.getString("icon"));
            }
            if(json.containsKey("system")){
                SystemDto system = ssoSystemService.getSystemByAlias(json.getString("system"));
                mume.setSystemId(system.getId());
            }
            if(json.containsKey(ResourceType.POWER_KEYS)){
                mume.setSsoRes(json.getString(ResourceType.POWER_KEYS));
            }
            return mume;
        }

        public PowerTreeCache getCache() {
            return cache;
        }

        public void setCache(PowerTreeCache cache) {
            this.cache = cache;
        }
    }


}
