package cn.com.duiba.sso.ui.mume;

import cn.com.duiba.sso.api.domain.dto.PowerDto;
import cn.com.duiba.sso.api.domain.dto.SystemDto;
import cn.com.duiba.sso.api.domain.enums.ResourceType;
import cn.com.duiba.sso.api.exception.SsoRunTimeException;
import cn.com.duiba.sso.api.service.power.PowerTreeCache;
import cn.com.duiba.sso.api.service.power.PowerTreeService;
import cn.com.duiba.sso.api.service.system.SsoSystemService;
import cn.com.duiba.sso.ui.SsoUIProperties;
import com.alibaba.fastjson.JSONObject;
import com.google.common.base.Joiner;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;

import java.util.List;
import java.util.Optional;
import java.util.function.Function;
import java.util.stream.Collectors;

public class SystemResourceMumeLoader implements MumeLoader {

    private static final String UI_SYSTEM_NAME = "zt-ui-boot";

    @Autowired
    private PowerTreeService powerTreeService;
    @Autowired
    private SsoSystemService ssoSystemService;
    @Autowired
    private SsoUIProperties ssoUIProperties;

    /**
     * 获取ui总菜单
     * 菜单项的配置是在 SSO系统 的权限——》资源管理 - 》中台UI基础项目
     *
     * 配置文档
     *
     * 资源名称：对应菜单名称
     * 资源类型：JSON
     * 资源内容：{
     *  "system":"sso-service",//归属的系统，会根据当前用户的系统权限进行校验
     *  "icon": "glyphicon-random",//一级菜单的图标，二级菜单不用配置
     *  "state": "app",//(必填)angular路由态 如果system 没有配置，该配置不会生效，强行根据url进行跳转
     *  "params":{id:1},//angular路由态的参数
     *  "url":"http://sso.duiba.com.cn/#!/system/systemList",(必填)菜单跳转链接,不需要指定域名，SSO会根据系统配置域名自动填充
     *  "keys":"app"//菜单权限校验的key  system没有指定无效
     * }
     * @return UI的菜单合集
     */
    @Override
    public List<SsoUIMume> loadUIMume() {

        String systemName = Optional.ofNullable(ssoUIProperties.getMenuResource()).orElse(UI_SYSTEM_NAME);
        SystemDto system = ssoSystemService.getSystemByAlias(systemName);

        PowerTreeCache cache = powerTreeService.getPowerTreeCache(system.getId());

        SsoUIMumeTransform transform = new SsoUIMumeTransform();
        transform.setCache(cache);

        return cache.getPowerTree(transform);
    }


    private class SsoUIMumeTransform implements Function<PowerDto,SsoUIMume> {

        private PowerTreeCache cache;

        private Joiner joiner = Joiner.on(".").skipNulls();

        @Override
        public SsoUIMume apply(PowerDto powerDto) {
            if(StringUtils.isBlank(powerDto.getJson())){
                throw new SsoRunTimeException("菜单项："+powerDto.getName()+"配置异常");
            }

            SsoUIMume mume = new SsoUIMume();
            mume.setId(powerDto.getId());
            mume.setParentId(powerDto.getParentId());
            mume.setName(powerDto.getName());

            JSONObject json = JSONObject.parseObject(powerDto.getJson());
            List<PowerDto> powerLinks = cache.getPowerWay(powerDto.getId());

            List<String> stateList = powerLinks.stream().map(power->{
                JSONObject item = JSONObject.parseObject(power.getJson());
                return item.getString("state");
            }).collect(Collectors.toList());

            mume.setUrl(json.getString("url"));
            mume.setState(joiner.join(stateList));
            if(json.containsKey("params")){
                mume.setParams(json.getJSONObject("params"));
            }
            if(json.containsKey("icon")){
                mume.setIcon(json.getString("icon"));
            }
            if(json.containsKey("system")){
                SystemDto system = ssoSystemService.getSystemByAlias(json.getString("system"));
                mume.setSystemId(system.getId());
            }
            if(json.containsKey(ResourceType.POWER_KEYS)){
                mume.setSsoRes(json.getString(ResourceType.POWER_KEYS));
            }
            return mume;
        }

        public PowerTreeCache getCache() {
            return cache;
        }

        public void setCache(PowerTreeCache cache) {
            this.cache = cache;
        }
    }
}
