package cn.com.duiba.biz.credits;

import cn.com.duiba.dao.HttpMessageDAO;
import cn.com.duiba.domain.HttpMessageDO;
import cn.com.duiba.order.center.api.dto.CreditsCallbackMessage;
import cn.com.duiba.order.center.api.dto.CreditsMessage;
import cn.com.duiba.service.ConsumerCreditsLogService;
import cn.com.duiba.service.CustomService;
import cn.com.duiba.service.HttpAsyncClientPool;
import cn.com.duiba.service.HttpRetryRulesService;
import cn.com.duiba.service.MessageService;
import cn.com.duiba.thirdparty.dto.CreditsMessageDto;
import cn.com.duiba.thirdparty.enums.CallbackChannelTypeEnum;
import cn.com.duiba.tool.AssembleTool;
import cn.com.duiba.tool.CaiNiaoTool;
import cn.com.duiba.tool.HttpRequestLog;
import cn.com.duiba.tool.JsonTool;
import cn.com.duiba.wolf.utils.BeanUtils;
import cn.com.duiba.wolf.utils.NumberUtils;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import org.apache.commons.collections.MapUtils;
import org.apache.http.Header;
import org.apache.http.HttpResponse;
import org.apache.http.client.entity.GzipDecompressingEntity;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.util.EntityUtils;
import org.apache.rocketmq.client.producer.DefaultMQProducer;
import org.apache.rocketmq.common.message.Message;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.nio.charset.Charset;
import java.util.Map;
import java.util.concurrent.ExecutorService;

/**
 * 加积分接口请求服务
 */
@Service
public class AddCreditsToDeveloper {

	private static final Logger LOG = LoggerFactory.getLogger(AddCreditsToDeveloper.class);

	@Autowired
	private HttpAsyncClientPool httpAsyncClientPool;

	@Autowired
	private MessageService messageService;

	@Autowired
	private HttpRetryRulesService httpRetryRulesService;

	@Autowired
	private HttpMessageDAO httpMessageDAO;

	@Autowired
	private DefaultMQProducer rocketMqProducer;

	@Autowired
	private CustomService customService;

	@Autowired
	private ConsumerCreditsLogService consumerCreditsLogService;

	@Resource(name = "httpCallbackExecutorService")
	private ExecutorService httpCallbackExecutorService;

	/**
	 * 提交加积分HTTP请求
	 * @param message
	 */
	public void submit(final HttpMessageDO message) {
        /** 用户定制加积分*/
        CreditsMessageDto creditsMessage = JsonTool.jsonToObject(message.getBizParams(), CreditsMessageDto.class);
		final CreditsMessageDto creditsMessageDto = customService.getRequestAddCredits(creditsMessage);

		HttpRequestBase http;
		if (CreditsMessage.HTTP_POST.equals(creditsMessageDto.getHttpType())) {
			Map<String, String> authParams = creditsMessageDto.getAuthParams();
			if (MapUtils.isNotEmpty(authParams)) {
				// 移除appSecret
				authParams.remove("appSecret");
			}
			http = AssembleTool.assembleRequest(creditsMessageDto.getHttpUrl(), authParams);
		} else {
			http = new HttpGet(creditsMessageDto.getHttpUrl());
		}

		HttpRequestLog.logUrl("[action addCredits] [tag request] [bizId " + creditsMessageDto.getRelationId() + "] [type " + creditsMessageDto.getRelationType() + "] [url " + creditsMessageDto.getHttpUrl() + "]");
       /** 定制header处理*/
		setHeaders(creditsMessageDto,http);

		httpAsyncClientPool.submit(creditsMessageDto.getAppId(), http, new FutureCallback<HttpResponse>() {

			@Override
			public void completed(HttpResponse response) {
				CreditsCallbackMessage resp = new CreditsCallbackMessage();
				try {
					resp.setCallbackType(CreditsCallbackMessage.CALLBACK_TYPE_COMPLETED);
					Header header = response.getEntity().getContentEncoding();
					String result;
					if (header != null && header.toString().contains(CaiNiaoTool.CONTENT_ENCODING_GZIP)) {
						result = EntityUtils.toString(new GzipDecompressingEntity(response.getEntity()));
					} else {
						result = EntityUtils.toString(response.getEntity());
					}
					String body = customService.getResponseCredits(NumberUtils.parseLong(creditsMessageDto.getAppId(), 0L), result);
					resp.setMessage(HttpRequestLog.subBody(body));
				} catch (Exception e) {
					LOG.error("toDeveloper completed", e);
				} finally {
					finallyBlock(message, creditsMessageDto, resp);
				}
			}

			@Override
			public void failed(Exception ex) {
				LOG.error("toDeveloper failed bizId:" + creditsMessageDto.getRelationId() + " bizType:" + creditsMessageDto.getRelationType(), ex);
				CreditsCallbackMessage resp = new CreditsCallbackMessage();
				try {
					resp.setCallbackType(CreditsCallbackMessage.CALLBACK_TYPE_FAILED);
					resp.setMessage(ex.getClass().getName() + ":" + ex.getMessage());
				} catch (Exception e) {
					LOG.error("toDeveloper failed", e);
				} finally {
					finallyBlock(message, creditsMessageDto, resp);
				}
			}

			@Override
			public void cancelled() {
				LOG.error("toDeveloper cancelled bizId:" + creditsMessageDto.getRelationId() + " bizType:" + creditsMessageDto.getRelationType());
				CreditsCallbackMessage resp = new CreditsCallbackMessage();
				resp.setCallbackType(CreditsCallbackMessage.CALLBACK_TYPE_FAILED);
				resp.setMessage("http cancelled");
				finallyBlock(message, creditsMessageDto, resp);
			}
		});


	}

	/**
	 * 定制 header处理
	 * */
	private void setHeaders(CreditsMessageDto message,HttpRequestBase http){
        //海底捞定制接口 做header处理
        Long appId = Long.valueOf(message.getAppId());
        if(customService.isHaiDiLao(appId)){
            CreditsMessage dto = BeanUtils.copy(message,CreditsMessage.class);
            customService.setHaidilaoSubAndAddHttpHeader(dto,http);
        }
    }

	/**
	 * 返回响应结果到业务方
	 * @param message
	 * @param req
	 * @param resp
	 */
	public void finallyBlock(HttpMessageDO message, CreditsMessageDto req, CreditsCallbackMessage resp) {
		HttpRequestLog.logUrl("[action addCredits] [tag response] [bizId " + req.getRelationId() + "] [type " + req.getRelationType() + "] [callback " + resp.getCallbackType() + "] [body " + resp.getMessage() + "]");
		if (nextRetry(message, resp)) {
			return;
		}
		try {
			resp.setRelationId(req.getRelationId());
			resp.setRelationType(req.getRelationType());
			resp.setParams(req.getParams());
			resp.setAppId(req.getAppId());
			resp.setConsumerId(req.getConsumerId());
			resp.setHttpUrl(req.getHttpUrl());
			String body = JsonTool.objectToJson(resp);
			String callbackTopic = req.getCallbackTopic();
			String callbackTag = req.getCallbackTag();
			String callbackKey = req.getCallbackKey();
			if(req.getParams() != null && CallbackChannelTypeEnum.ROCKETMQ.getType().equals(req.getParams().get(CreditsMessageDto.MESSAGE_CHANNEL_TYPE_KEY))){
				sendRocketMQMessage(callbackTopic, callbackTag, callbackKey, body);
			} else {
				messageService.sendMsg(callbackTopic, callbackTag, callbackKey, body);
			}

			saveCreditsLog(req, resp);
		} catch (Exception e) {
			LOG.error("credits callback: bizId:" + req.getRelationId() + " bizType:" + req.getRelationType(), e);
		}
	}

	/**
	 * 判断是否需要重试<br/>
	 * 1.记录重试数据<br/>
	 * 2.定时扫描<br/>
	 * 
	 * @param message
	 * @param resp
	 * @return true 下次重试， false 无需重试
	 */
	public boolean nextRetry(HttpMessageDO message, CreditsCallbackMessage resp) {
		try {
			// 如果响应包含OK认为是成功
			if (isSuccess(resp)) {
				httpMessageDAO.delete(message.getId());
				return false;
			}
			// 是否超过重试次数
			if (message.getNumber() >= httpRetryRulesService.getRetryNumber(message)) {
				httpMessageDAO.delete(message.getId());
				return false;
			}
			// 更新下次重试时间
			httpRetryRulesService.updateNextTime(message);
			return true;
		} catch (Exception e) {
			LOG.error("nextRetry", e);
		}
		return false;
	}

	/**
	 * 解析响应的内容是否为成功
	 * 
	 * @param resp
	 * @return
	 */
	public boolean isSuccess(CreditsCallbackMessage resp) {
		try {
			if (CreditsCallbackMessage.CALLBACK_TYPE_COMPLETED.equals(resp.getCallbackType())) {
				String body = resp.getMessage().toLowerCase();
				if (body.contains("ok") || body.contains("success")) {
					return true;
				}
			}
			return false;
		} catch (Exception e) {
			LOG.error("isSuccess", e);
			return true;
		}
	}

	private void sendRocketMQMessage(String topic, String tag, String key, String message){
		Message msg = new Message(topic, tag, key, message.getBytes(Charset.forName("utf-8")));
		try {
			rocketMqProducer.send(msg);
		} catch (Exception e) {
			LOG.error("addCredits sendRocketMQMessage", e.getMessage());
		}
	}

	/**
	 * 保存积分记录
	 * @param body
	 * @param request
	 * @param response
	 */
	private void saveCreditsLog(CreditsMessageDto request, CreditsCallbackMessage response) {
		httpCallbackExecutorService.execute(() -> {
			try {
				if (CreditsCallbackMessage.CALLBACK_TYPE_COMPLETED.equals(response.getCallbackType())) {
					JSONObject json = JSON.parseObject(response.getMessage());
					if ("success".equalsIgnoreCase(json.getString("status")) || "ok".equalsIgnoreCase(json.getString("status"))) {
						consumerCreditsLogService.save(request, response);
					}
				}
			} catch (Exception e) {
				LOG.error("积分记录保存失败, bizId={}, bizType={}", request.getRelationId(), request.getRelationType(), e);
			}
		});
	}

}
