package cn.com.duiba.biz.credits;

import cn.com.duiba.boot.exception.BizException;
import cn.com.duiba.constant.BeingmateConfig;
import cn.com.duiba.domain.SupplierRequest;
import cn.com.duiba.tool.AssembleTool;
import cn.com.duiba.tool.JsonTool;
import com.alibaba.fastjson.JSONObject;
import com.github.benmanes.caffeine.cache.Caffeine;
import com.google.common.collect.Maps;
import org.apache.commons.lang.StringUtils;
import org.apache.http.client.HttpClient;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.impl.conn.PoolingHttpClientConnectionManager;
import org.apache.http.util.EntityUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.io.IOException;
import java.nio.charset.Charset;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.TimeUnit;

/**
 * @program: thirdparty-all
 * @description: 贝因美api
 * @author: Simba
 * @create: 2020-01-07 17:33
 **/
@Service
public class BeingmateApi {
    private static final Logger LOGGER = LoggerFactory.getLogger(BeingmateApi.class);

    @Autowired
    private BeingmateConfig beingmateConfig;

    /**
     * 存放token的缓存,token 60分钟失效，设置失效时间50分钟，避免临界时间调用
     */
    private com.github.benmanes.caffeine.cache.Cache<String, String> tokenCache = Caffeine.newBuilder()
            .expireAfterWrite(50, TimeUnit.MINUTES)
            .initialCapacity(5)
            .maximumSize(100)
            .build();

    private static final String TOKEN = "token";

    /**
     * 获取虚拟商品请求
     *
     * @param request
     * @return
     */
    public SupplierRequest getVirtualRequest(SupplierRequest request) {
        String url = request.getHttpUrl();
        String params = url.substring(url.indexOf('?') + 1);
        // 1.将请求URL的参数转换为MAP
        Map<String, String> paramMap = AssembleTool.getUrlParams(params);
        Map<String, String> authParams = request.getAuthParams();
        authParams.put("Openid", paramMap.get("uid"));
        authParams.put("SerialNumber", paramMap.get("orderNum"));
        authParams.put("postType", "raw");
//        authParams.put("Openid", "OPENID_CESHI");
//        authParams.put("SerialNumber", "SerialNumber_ceshi12");
        authParams.put("unicodeType", "UTF-8");
        request.setAuthParams(authParams);
        return request;
    }

    /**
     * 设置请求头
     *
     * @param http
     */
    public void setHttpHeader(HttpRequestBase http) {
        //1、获取token
        String token = tokenCache.get(TOKEN, key -> {
            String tokenTemp = getToken();
            if (StringUtils.isBlank(tokenTemp)) {
                return null;
            }
            return tokenTemp;
        });
        //3、设置请求头
        http.setHeader("Content-Type", "application/json");
        http.setHeader("charset", "UTF-8");
        http.setHeader("Authorization", "Bearer " + token);
    }

    /**
     * 获取虚拟商品响应信息
     *
     * @param message
     * @param body
     * @return
     */
    public String getVirtualResponse(SupplierRequest message, String body) {
        Map<String, String> duibaDoc = new HashMap<>();
        try {
            JSONObject resultJson = JSONObject.parseObject(body);
            if (null == resultJson) {
                throw new BizException("开发者接口响应内容异常");
            }
            Boolean success = resultJson.getBoolean("success");
            String status = resultJson.getString("status");
            if (!Objects.equals("200", status) || !success) {
                throw new BizException(resultJson.getString("msg"));
            }
            duibaDoc.put("status", "success");
            duibaDoc.put("data", body);
        } catch (Exception e) {
            duibaDoc.put("status", "fail");
            duibaDoc.put("errorMessage", e.getMessage());
            LOGGER.info("[BeingmateApi-getVirtualResponse] Params:[message={},body={}], Msg:贝因美-虚拟商品接口响应解析异常", JSONObject.toJSONString(message), body, e);
        }
        return JsonTool.objectToJson(duibaDoc);
    }

    /**
     * 获取请求token，60分钟失效
     *
     * @return
     */
    private String getToken() {
        String result = null;
        try {
            Map<String, String> paramMap = Maps.newHashMap();
            paramMap.put("appid", beingmateConfig.getAppKey());
            paramMap.put("secret", beingmateConfig.getAppSecret());
            String url = AssembleTool.assembleUrl(beingmateConfig.getJwttokenUrl(), paramMap);

            result = sendGet(createHttpClient(), url, Charset.forName("utf-8"));

            /**
             * 返回示例值
             * {
             *   "success": true,
             *   "msg": "",
             *   "response": {
             *     "token": "eyJhbGciOiJIUzI1NiIsInR5cCI6IkpXVCJ9.eyJodHRwOi8vc2NoZW1hcy54bWxzb2FwLm9yZy93cy8yMDA1LzA1L2lkZW50aXR5L2NsYWltcy9uYW1lIjoiZHVpYmEiLCJqdGkiOiI0NCIsImh0dHA6Ly9zY2hlbWFzLm1pY3Jvc29mdC5jb20vd3MvMjAwOC8wNi9pZGVudGl0eS9jbGFpbXMvZXhwaXJhdGlvbiI6IjEvNy8yMDIwIDY6NTE6NTMgUE0iLCJodHRwOi8vc2NoZW1hcy5taWNyb3NvZnQuY29tL3dzLzIwMDgvMDYvaWRlbnRpdHkvY2xhaW1zL3JvbGUiOiJkdWliYSIsIm5iZiI6MTU3ODM5MDcxMywiZXhwIjoxNTc4Mzk0MzEzLCJpc3MiOiJieW0uQ29yZSIsImF1ZCI6IndyIn0.eVj2I8_p6cH3MuD-KAx76ndADJctetO-GyqUE-46XnQ",
             *     "expires_in": 3600,
             *     "token_type": "Bearer"
             *   },
             *   "status": "200"
             * }
             */
            JSONObject resultJson = JSONObject.parseObject(result);
            if ((boolean) resultJson.get("success") && Objects.equals("200", resultJson.get("status"))) {
                JSONObject response = JSONObject.parseObject(resultJson.getString("response"));
                return response.getString("token");
            }
        } catch (Exception e) {
            LOGGER.warn("贝因美获取token值异常, result={}", result, e);
        }
        LOGGER.warn("贝因美获取token结果为={}", result);
        return null;
    }

    private static HttpClient createHttpClient() {
        RequestConfig defaultRequestConfig = RequestConfig.custom().setSocketTimeout(5000).setConnectTimeout(5000).setConnectionRequestTimeout(3000).build();
        PoolingHttpClientConnectionManager cm = new PoolingHttpClientConnectionManager();
        cm.setMaxTotal(200);
        cm.setDefaultMaxPerRoute(20);
        return HttpClients.custom().setConnectionManager(cm).setDefaultRequestConfig(defaultRequestConfig).build();
    }

    private static String sendGet(HttpClient httpClient, String url, Charset encoding) {
        HttpGet httpget = new HttpGet(url);
        String resp = "";
        // 执行get请求.
        CloseableHttpResponse response = null;
        try {
            response = (CloseableHttpResponse) httpClient.execute(httpget);
            resp = EntityUtils.toString(response.getEntity(), encoding);
        } catch (IOException e) {
            LOGGER.warn("贝因美获取token发送get请求失败，url={}", url, e);
        } finally {
            if (response != null) {
                try {
                    response.close();
                } catch (IOException e) {
                    LOGGER.warn("贝因美获取token关闭response失败", e);
                }
            }
        }
        return resp;
    }
}

