package cn.com.duiba.biz.credits;

import cn.com.duiba.biz.Exception.ThirdpatyException;
import cn.com.duiba.constant.UnionPayConstants;
import cn.com.duiba.constant.ZJUnionBankConstants;
import cn.com.duiba.domain.SupplierRequest;
import cn.com.duiba.tool.AssembleTool;
import cn.com.duiba.tool.DES3Tool;
import cn.com.duiba.tool.ZjUnionBankUtil;
import cn.hutool.core.date.DateUtil;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import java.security.KeyFactory;
import java.security.PrivateKey;
import java.security.Signature;
import java.security.spec.PKCS8EncodedKeySpec;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Random;
import org.apache.commons.codec.binary.Base64;
import org.apache.commons.lang3.StringEscapeUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.entity.ByteArrayEntity;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.StringEntity;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

/**
 * 云闪付定制接口
 *
 * @author XuJing
 * @since 2020/10/26 10:04 上午
 */
@Service
public class UnionPayApi {

	private static final Logger LOGGER = LoggerFactory.getLogger(UnionPayApi.class);

	@Autowired
	private UnionPayConstants unionPayConstants;

	@Autowired
	private ZJUnionBankConstants zjUnionBankConstants;

	/**
	 * 判断当前app是否是云闪付
	 *
	 * @param appId
	 * @return
	 */
	public boolean isUnionPayApp(Long appId) {
		return unionPayConstants.getAppConfig(String.valueOf(appId)) != null;
	}

	/**
	 * 拼接云闪付积分兑换参数
	 *
	 * @param request
	 * @return
	 */
	public HttpRequestBase getVirtualRequest(SupplierRequest request) {
		String url = request.getHttpUrl();
		String authParams = url.substring(url.indexOf('?') + 1);
		Map<String, String> authParamMap = AssembleTool.getUrlParams(authParams);
		String goodsNum = authParamMap.get("params");
		LOGGER.info("银联url:{}, goodNum:{}",url,goodsNum);

		if (StringUtils.isBlank(goodsNum) || !checkGoodsPrefix(goodsNum)) {
			return new HttpGet(url);
		}
		Map<String, String> requestParams = buildVirtualRequestParams(authParamMap, request.getParams(), Long.valueOf(request.getAppId()));
		LOGGER.info("银联虚拟商品请求参数:{}",JSON.toJSONString(requestParams));

		if (goodsNum.startsWith(zjUnionBankConstants.getVirtualCreditsPre())) {
			// 云闪付 - 通用券平台
			ZjUnionBankHelper zjUnionBankHelper = new ZjUnionBankHelper();
			return zjUnionBankHelper.getVirtualRequest(request);
		}

		HttpPost post = new HttpPost(unionPayConstants.getVirtualUrl());
		post.setEntity(new StringEntity(JSON.toJSONString(requestParams), ContentType.APPLICATION_JSON));
		return post;
	}

	/**
	 * 构造虚拟商品请求参数
	 */
	private Map<String, String> buildVirtualRequestParams(Map<String, String> authParamMap, Map<String,String> params, Long appId) {
		Map<String, String> requestParams = new HashMap<>();
		//获取银联appId
		if(StringUtils.isBlank(params.get("devAppId"))){
			requestParams.put("appId", unionPayConstants.getAppConfig(String.valueOf(appId)).getAppId());
		}else{
			requestParams.put("appId", params.get("devAppId"));
		}

		requestParams.put("transSeqId", authParamMap.get("orderNum"));
		requestParams.put("transTs", DateUtil.format(new Date(), "yyyyMMdd"));
		requestParams.put("couponId", authParamMap.get("params").replaceAll(unionPayConstants.getPrefix(), ""));
		if(StringUtils.isBlank(params.get("devAppId"))){
			requestParams.put("openId", authParamMap.get("uid"));
		}else{
			requestParams.put("mobile", params.get("mobile"));
		}
		requestParams.put("acctEntityTp", "03");
		requestParams.put("couponNum", "1");
		requestParams.put("nonceStr", createNonceStr());
		requestParams.put("timestamp", Long.toString(System.currentTimeMillis() / 1000));
		requestParams.put("signature", sign(requestParams, unionPayConstants.getPrivateKey()));
		if(StringUtils.isNotBlank(params.get("devAppId"))){
			try {
				LOGGER.info("appId:{},devAppId:{},SymmetricKey:{}", appId, params.get("devAppId"), unionPayConstants.getSymmetricKey(params.get("devAppId")));
				requestParams.put("mobile", DES3Tool.getEncryptedValue(params.get("mobile"),unionPayConstants.getSymmetricKey(params.get("devAppId"))));
			} catch (Exception e) {
				LOGGER.warn("银联虚拟商品请求参数加密失败, mobile: =>{}, SymmetricKey: =>{}",params.get("mobile"),
						unionPayConstants.getSymmetricKey(params.get("devAppId")),e);
			}
		}
		return requestParams;
	}

	/**
	 * 解析云闪付积分兑换请求结果
	 *
	 * @param message
	 * @param body
	 * @return
	 */
	public String getVirtualResponse(SupplierRequest message, String body) {
		String url = message.getHttpUrl();
		String authParams = url.substring(url.indexOf('?') + 1);
		Map<String, String> authParamMap = AssembleTool.getUrlParams(authParams);
		String goodsNum = authParamMap.get("params");

		if (StringUtils.isBlank(goodsNum) || !checkGoodsPrefix(goodsNum)) {
			return body;
		}

		if (goodsNum.startsWith(zjUnionBankConstants.getVirtualCreditsPre())) {
			// 云闪付 - 通用券平台
			ZjUnionBankHelper zjUnionBankHelper = new ZjUnionBankHelper();
			return zjUnionBankHelper.getVirtualResponse(message, body);
		}

		Map<String, String> map = new HashMap<>();
		if (StringUtils.isBlank(body)) {
			throw new ThirdpatyException("云闪付，虚拟商品定制，返回结果为空");
		}
		LOGGER.info("云闪付定制返回结果为:{}", body.length() > 500 ? body.substring(0, 500): body);
		try {
			JSONObject jsonObject = JSON.parseObject(body);
			map.put("status", Objects.equals("00", jsonObject.getString("resp")) ? "success" : "fail");
			map.put("errorMessage", StringEscapeUtils.unescapeHtml4(jsonObject.getString("msg")));
			map.put("supplierBizId", jsonObject.getString("transSeqId"));
		} catch (Exception e) {
			LOGGER.error("云闪付，虚拟商品定制，结果解析错误:{}", body, e);
			map.put("status", "fail");
			map.put("errorMessage", "虚拟商品充值接口响应解析错误");
		}
		return JSON.toJSONString(map);
	}

	/**
	 * 生成签名随机值
	 *
	 * @return
	 */
	public String createNonceStr() {
		String sl = "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789";
		StringBuilder sb = new StringBuilder();
		for (int i = 0; i < 16; i++) {
			sb.append(sl.charAt(new Random().nextInt(sl.length())));
		}
		return sb.toString();
	}


	/**
	 * 云闪付RSA签名
	 */
	public String sign(Map<String, String> param, String signKey) {
		try {
			String value = sortMap(param);
			LOGGER.info("云闪付定制签名原串为:{}", value);
			byte[] keyBytes = Base64.decodeBase64(signKey);
			PKCS8EncodedKeySpec keySpec = new PKCS8EncodedKeySpec(keyBytes);
			KeyFactory keyf = KeyFactory.getInstance("RSA");
			PrivateKey priKey = keyf.generatePrivate(keySpec);
			Signature signature = Signature.getInstance("SHA256WithRSA");
			signature.initSign(priKey);
			signature.update(value.getBytes());
			byte[] signed = signature.sign();
			return Base64.encodeBase64String(signed);
		} catch (Exception e) {
			LOGGER.error("云闪付定制签名失败", e);
			throw new ThirdpatyException("云闪付定制签名失败");
		}
	}


	/**
	 * 参数名按ASCII码从小到大排序（字典序）
	 *
	 * @return
	 */
	public static String sortMap(Map<String, String> param) {
		StringBuilder result = new StringBuilder();
		Collection<String> keySet = param.keySet();
		List<String> list = new ArrayList<String>(keySet);
		Collections.sort(list);
		for (int i = 0; i < list.size(); ++i) {
			String key = list.get(i);
			if ("symmetricKey".equals(key)) {
				continue;
			}
			result.append(key).append("=").append(param.get(key)).append("&");
		}
		String sortMap = result.substring(0, result.length() - 1);
		return sortMap;
	}

	/**
	 * 检查商品前缀
	 * @param goods
	 * @return
	 */
	private boolean checkGoodsPrefix(String goods) {
		return getPrefixList().stream().anyMatch(goods::startsWith);
	}

	/**
	 * 获取所有前缀列表
	 * @return
	 */
	private List<String> getPrefixList() {
		return Arrays
			.asList(unionPayConstants.getPrefix(), zjUnionBankConstants.getVirtualCreditsPre());
	}

	/**
	 * 通用券平台
	 */
	class ZjUnionBankHelper {

		/**
		 * 虚拟商品请求
		 *
		 * @param request
		 */
		public HttpRequestBase getVirtualRequest(SupplierRequest request) {
			LOGGER.info("浙江银联虚拟商品请求, request={}", JSON.toJSONString(request));

			String url = request.getHttpUrl();
			Map<String, String> params = request.getParams();
			String paramsStr = url.substring(url.indexOf('?') + 1);
			Map<String, String> authParams = AssembleTool.getUrlParams(paramsStr);
			String goodsNum = authParams.get("params");

			if (params == null || params.get("mobile") == null) {
				LOGGER.warn("浙江银联虚拟兑换接口参数错误 param:{}", JSON.toJSONString(params));
				throw new RuntimeException("浙江银联虚拟兑换接口参数错误");
			}

			if (StringUtils.isBlank(goodsNum) || !goodsNum.startsWith(zjUnionBankConstants.getVirtualCreditsPre())) {
				LOGGER.warn("浙江银联未定制虚拟兑换接口 param:{}",JSON.toJSONString(authParams));
				throw new RuntimeException("浙江银联未定制虚拟兑换接口");
			}
			goodsNum = goodsNum.replace(zjUnionBankConstants.getVirtualCreditsPre(), "");
			String[] goodsArray = goodsNum.split("_");
			if (goodsArray.length != 2) {
				LOGGER.warn("浙江银联商品配置错误 param:{}",JSON.toJSONString(authParams));
				throw new RuntimeException("浙江银联商品配置错误");
			}

			String mobile = params.get("mobile");
			String couponModelId = goodsArray[0];
			String issueChannelId = goodsArray[1];

			JSONObject object = new JSONObject();
			object.put("mobile_phone_num", mobile);

			JSONObject privateInfo = new JSONObject();
			privateInfo.put("sysId", "CXCP");
			try {
				privateInfo.put("data", ZjUnionBankUtil
					.encerypt(object.toJSONString(), zjUnionBankConstants.getPublicKey()));
			} catch (Exception e) {
				LOGGER.warn("浙江银联商品创建加密错误 param:{}",JSON.toJSONString(authParams));
				throw new RuntimeException("浙江银联商品创建失败");
			}

			JSONObject transferData = new JSONObject();
			transferData.put("payType", 3);
			HashMap<String, Object> map = new HashMap<>();
			map.put("order_no", request.getOrderId());
			map.put("coupon_model_id", couponModelId);
			map.put("issue_channel_id", issueChannelId);
			map.put("transfer_data", transferData.toJSONString());
			map.put("is_pay", 1);
			map.put("private_info", privateInfo.toJSONString());

			return getHttpRequestBase(map, zjUnionBankConstants.getPropSvcApi());
		}

		/**
		 * 解析虚拟商品充值响应结果
		 *
		 * @param request
		 * @param body
		 */
		public String getVirtualResponse(SupplierRequest request, String body) {
			LOGGER.info("浙江银联虚拟商品响应结果response, body={}， {}", body, JSONObject.toJSONString(request));

			body = verifySign(body);

			if ("false".equals(body)) {
				throw new RuntimeException("浙江银联银行未定制虚拟兑换, 接口响应 解析异常");
			}

			JSONObject jsonObject = JSONObject.parseObject(body);

			JSONObject result = new JSONObject();
			result.put("status", "success");
			if (jsonObject.getInteger("status") != 1) {
				result.put("status", "fail");
				result.put("errorMessage", "虚拟商品发放失败");
			}

			return result.toString();
		}

		private String verifySign(String body) {
			LOGGER.info("浙江银联虚拟商品验签, {}", body);
			JSONObject parse = JSONObject.parseObject(body);
			Object respCd = parse.get("respCd");
			if (!"0000".equals(respCd.toString())) {
				String respMsg = parse.get("respMsg").toString();
				LOGGER.info("浙江银联-响应结果异常 {}, {}", respCd, respMsg);
				return "false";
			}

			// 校验签名
			String sign = parse.get("sign").toString();
			try {
				if (StringUtils.isBlank(sign) || !ZjUnionBankUtil.verifySign(parse, sign, zjUnionBankConstants.getPublicKey())) {
					LOGGER.info("浙江银联-验签失败 {}", body);
					return "false";
				}
			} catch (Exception exception) {
				LOGGER.info("浙江银联-验签异常 {}，{}", body, exception.getMessage());
				return "false";
			}

			return parse.get("respContent").toString();
		}

		/**
		 * @param paramMap
		 * @param svcApi
		 * @return
		 */
		private HttpRequestBase getHttpRequestBase(Map<String, Object> paramMap, String svcApi) {
			String url = zjUnionBankConstants.getUrl();
			JSONObject reqMsg = new JSONObject();
			reqMsg.put("svcId", zjUnionBankConstants.getSvcId());
			reqMsg.put("svcApi", svcApi);
			reqMsg.put("signType", "RSA2");
			reqMsg.put("version", "1.0.0");
			reqMsg.put("timestamp", (new SimpleDateFormat("yyyyMMddHHmmss")).format(new Date()));
			reqMsg.put("bizContent", JSON.toJSONString(paramMap));

			LOGGER.info("浙江银联调用服务：" + url + "   原请求报文：" + reqMsg.toJSONString());

			HttpPost httpPost = null;
			try {
				reqMsg.put("sign", ZjUnionBankUtil.sign(reqMsg, zjUnionBankConstants.getPriKey()));
				httpPost = new HttpPost(url);
				ByteArrayEntity bae = new ByteArrayEntity(reqMsg.toJSONString().getBytes());
				httpPost.setEntity(bae);
				httpPost.addHeader("Content-Type", "application/json; charset=UTF-8");
			} catch (Exception e) {
				LOGGER.warn("浙江银联 getHttpRequestBase", e);
			}
			return httpPost;
		}
	}
}
