package cn.com.duiba.biz.credits.strategy.Impl;

import cn.com.duiba.biz.credits.strategy.ApiStrategy;
import cn.com.duiba.boot.exception.BizException;
import cn.com.duiba.constant.WandaConfig;
import cn.com.duiba.domain.SubCreditsMsgWrapper;
import cn.com.duiba.domain.SupplierRequest;
import cn.com.duiba.enums.wanda.WandaErrorCode;
import cn.com.duiba.enums.wanda.WandaRequestStatus;
import cn.com.duiba.notifycenter.domain.NotifyQueueDO;
import cn.com.duiba.thirdparty.dto.CreditsMessageDto;
import cn.com.duiba.tool.AssembleTool;
import cn.com.duiba.wolf.utils.UrlUtils2;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.google.common.collect.Maps;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang3.RandomStringUtils;
import org.apache.http.NameValuePair;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.entity.UrlEncodedFormEntity;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.message.BasicNameValuePair;
import org.apache.http.util.EntityUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.nio.charset.StandardCharsets;
import java.util.*;
import java.util.stream.Collectors;

/**
 * @author fja
 * @since 2021-04-19
 */
@Service
public class WandaApiStrategy implements ApiStrategy {

    private static final String COMPANY_CODE = "companyCode";

    private static final String GRANT_TYPE = "grantType";

    private static final String ACCOUNT = "account";

    private static final String PASSWORD = "password";

    private static final String ACCESS_TOKEN = "accessToken";

    private static final String SUCCESS = "Success";

    private static final long SUCCESS_CODE = 200;

    private static final String RESPONSE_CODE = "responseCode";

    private static final String VIRTUAL = "virtual";

    private static final String TRUE = "true";

    private final Logger LOGGER = LoggerFactory.getLogger(this.getClass());

    @Resource
    private WandaConfig wandaConfig;

    @Resource(name = "httpClient")
    private CloseableHttpClient httpClient;

    private final RequestConfig config;
    private static final int FIVE_SECONDS = 5*1000;

    private static final int TEN_SECONDS = 10*1000;

    {
        //http请求超时配置
        config = RequestConfig.custom().setConnectTimeout(FIVE_SECONDS).setSocketTimeout(TEN_SECONDS).setConnectionRequestTimeout(500).build();
    }

    @Override
    public HttpRequestBase getMqSubCreditsHttpRequest(SubCreditsMsgWrapper message) {
        try {
            String accessToken = getToken();
            LOGGER.info("[万达酒店]接口授权accessToken = [{}]", accessToken);
            //请求链接
            String url = getHostName(message.getHttpUrl());
            HttpPost httpPost = new HttpPost(url);
            Map<String, String> originData = AssembleTool.getUrlParams(getParamUrl(message.getHttpUrl()));
            LOGGER.info("[万达酒店]减积分originData = [{}]", JSON.toJSONString(originData));
            Map<String, Object> params = Maps.newHashMap();
            //设置请求头
            httpPost.setHeader("Authentication", accessToken);
            //请求参数
            //卡号（partnerUserId就是卡号）
            params.put("cardNo", originData.get("uid"));
            //订单号
            params.put("orderNumber", originData.get("orderNum"));
            //积分
            params.put("points", Integer.parseInt(originData.get("credits")));
            //交易流水号，撤销积分会使用到，这里使用兑吧订单号
            params.put("transactionNumber", originData.get("orderNum"));
            //类型
            String type = Optional.ofNullable(originData.get("type")).orElse("");
            //是否是兑吧商品
            Map<String, String> data = Optional.ofNullable(message.getSubCreditsMsg().getParams()).orElse(Maps.newHashMap());
            String isDuibaItem = data.get("isDuibaItem");
            //itemCode商家商品号，酒店编码默认
            //String storeCode = getStoreCode(originData.get("itemCode"), type, TRUE.equals(isDuibaItem));
            params.put("storeCode", wandaConfig.getDefaultStoreCode());
            //备注
            params.put("remarks", originData.get("description"));
            LOGGER.info("[万达酒店]扣减积分参数 = [{}]", JSON.toJSONString(params));
            httpPost.setEntity(new StringEntity(JSON.toJSONString(params), ContentType.APPLICATION_JSON));
            httpPost.setConfig(config);
            message.setHttpUrl(url);
            message.getSubCreditsMsg().setAuthParams(params.entrySet().stream().collect(Collectors.toMap(Map.Entry::getKey, entry -> String.valueOf(entry.getValue()))));
            return httpPost;
        } catch (Exception e) {
            LOGGER.info("[万达酒店]扣减积分请求生成失败", e);
            throw new IllegalStateException(e);
        }
    }

    @Override
    public HttpRequestBase getAddCreditsMessageRequest(CreditsMessageDto message) {
        try {
            String accessToken = getToken();
            LOGGER.info("[万达酒店]接口授权accessToken = [{}]", accessToken);
            Map<String, String> originData = AssembleTool.getUrlParams(getParamUrl(message.getHttpUrl()));
            LOGGER.info("[万达酒店]加积分originData = [{}]", JSON.toJSONString(originData));
            String url = getHostName(message.getHttpUrl());
            HttpPost httpPost = new HttpPost(url);
            Map<String, Object> params = Maps.newHashMap();
            //设置请求头
            httpPost.setHeader("Authentication", accessToken);
            //请求参数
            //卡号（partnerUserId就是卡号）
            params.put("cardNo", originData.get("uid"));
            //订单号
            params.put("orderNumber", originData.get("orderNum"));
            //积分
            params.put("points", Integer.parseInt(originData.get("credits")));
            //酒店编码（默认编码）
            params.put("storeCode", wandaConfig.getDefaultStoreCode());
            //备注
            params.put("remarks", originData.get("description"));
            //交易流水号，撤销积分会使用到，这里使用兑吧订单号
            params.put("transactionNumber", originData.get("orderNum"));
            LOGGER.info("[万达酒店]加积分请求参数 = [{}]", JSON.toJSONString(params));
            httpPost.setEntity(new StringEntity(JSON.toJSONString(params), ContentType.APPLICATION_JSON));
            httpPost.setConfig(config);
            message.setHttpUrl(url);
            message.setAuthParams(params.entrySet().stream().collect(Collectors.toMap(Map.Entry::getKey, entry -> String.valueOf(entry.getValue()))));
            return httpPost;
        } catch (Exception e) {
            LOGGER.info("[万达酒店]加积分请求生成失败", e);
            throw new IllegalStateException(e);
        }
    }


    /**
     * 解析返回值
     *
     * @param body       回包
     * @param addCredits 积分
     * @param authParams 数据
     * @return parseCreditsResponse
     */
    @Override
    public String parseCreditsRsp(String body, Boolean addCredits, Map<String, String> authParams) {
        JSONObject responseBody = JSON.parseObject(body);
        LOGGER.info("[万达酒店]积分返回body=[{}]", body);
        JSONObject result = new JSONObject();
        if (Objects.equals(SUCCESS, responseBody.getString(RESPONSE_CODE))) {
            result.put("status", "ok");
            // 开发者不返回订单号，自定义
            result.put("bizId", System.currentTimeMillis() + RandomStringUtils.randomNumeric(6));
        } else {
            result.put("status", "fail");
            result.put("errorMessage", responseBody.getString("message"));
        }
        try {
            LOGGER.info("[万达酒店] parseCreditsRsp#authParams = {}", JSON.toJSONString(authParams));
            Long credits = queryCredits(authParams.get("cardNo"));
            result.put("credits", credits);
        } catch (Exception e) {
            LOGGER.error("[万达酒店] 查询用户积分失败", e);
            return result.toString();
        }
        return result.toString();
    }

    @Override
    public HttpRequestBase getVirtualRequest(SupplierRequest request) {
        try {
            String accessToken = getToken();
            LOGGER.info("[万达酒店]接口授权accessToken = [{}]", accessToken);
            String url = UrlUtils2.extractUrl(request.getHttpUrl());
            Map<String, String> originData = UrlUtils2.extractUrlParamsFromUrl(request.getHttpUrl());
            LOGGER.info("[万达酒店]虚拟商品兑换originData = [{}]", JSON.toJSONString(originData));
            HttpPost httpPost = new HttpPost(url);
            Map<String, String> params = Maps.newHashMap();
            //设置请求头
            httpPost.setHeader("Authentication", accessToken);
            //请求参数
            //卡号（partnerUserId就是卡号）
            params.put("cardNo", originData.get("uid"));
            // 兑换券编码
            final String splitter = "_";
            String codeAndStoreCode = originData.get("params");
            if (StringUtils.isNotBlank(codeAndStoreCode) && codeAndStoreCode.contains(splitter)) {
                String[] splitStrs = codeAndStoreCode.split(splitter);
                params.put("code", splitStrs[0]);
            } else {
                throw new IllegalStateException("虚拟商品商家编码格式异常[" + codeAndStoreCode + "]");
            }
            // 兑换数量
            params.put("count", "1");
            //交易流水号，这里使用兑吧订单号
            params.put("transactionNumber", originData.get("orderNum"));
            LOGGER.info("[万达酒店]虚拟商品兑换参数 = [{}]", JSON.toJSONString(params));
            httpPost.setEntity(new StringEntity(JSON.toJSONString(params), ContentType.APPLICATION_JSON));
            httpPost.setConfig(config);
            request.setAuthParams(params);
            request.setHttpUrl(url);
            return httpPost;
        } catch (Exception e) {
            LOGGER.info("[万达酒店]虚拟商品兑换请求生成失败 orderId=" + request.getOrderId(), e);
            throw new IllegalStateException(e);
        }
    }

    @Override
    public String getVirtualResponse(SupplierRequest request, String body) {
        JSONObject result = new JSONObject();
        try {
            JSONObject responseJson = JSONObject.parseObject(body);
            LOGGER.info("[万达酒店]虚拟商品兑换body=[{}]", body);
            Long responseCode = Optional.ofNullable(responseJson.getLong("code")).orElse(-1L);
            if (responseCode == SUCCESS_CODE) {
                result.put("status", "success");
            } else {
                result.put("status", "fail");
                result.put("errorMessage", responseJson.getString("message"));
            }
        } catch (Exception e) {
            result.put("status", "fail");
            result.put("errorMessage", e.getMessage());
            LOGGER.info("[万达酒店]解析虚拟商品兑换请求结果异常 orderId=" + request.getOrderId(), e);
        }
        return result.toJSONString();
    }

    @Override
    public HttpRequestBase getRequestNotify(String notifyUrl, NotifyQueueDO record) {

        if (record.getResult()) {
            throw new IllegalStateException("兑换成功不能回退积分");
        }

        try {
            String accessToken = getToken();
            LOGGER.info("[万达酒店]接口授权accessToken = [{}]", accessToken);
            HttpPost httpPost = new HttpPost(notifyUrl);
            Map<String, String> params = Maps.newHashMap();
            //设置请求头
            httpPost.setHeader("Authentication", accessToken);
            //请求参数
            //卡号（partnerUserId就是卡号）
            params.put("cardNo", record.getPartnerUserId());
            //订单号
            params.put("orderNumber", record.getDuibaOrderNum());
            //交易流水号，这里使用兑吧订单号
            params.put("transactionNumber", record.getDuibaOrderNum());
            LOGGER.info("[万达酒店]积分回退参数 = [{}]", JSON.toJSONString(params));
            httpPost.setEntity(new StringEntity(JSON.toJSONString(params), ContentType.APPLICATION_JSON));
            httpPost.setConfig(config);
            return httpPost;
        } catch (Exception e) {
            LOGGER.info("[万达酒店]积分回退请求生成失败 orderId=" + record.getDuibaOrderNum().replace("C", ""), e);
            throw new IllegalStateException(e);
        }
    }

    @Override
    public String getResponseNotify(String body) {
        try {
            JSONObject responseJson = JSONObject.parseObject(body);
            String responseCode = responseJson.getString("responseCode");
            if (SUCCESS.equals(responseCode)) {
                return "ok";
            }
        } catch (Exception e) {
            LOGGER.info("[万达酒店]解析积分回退请求结果异常", e);
        }
        return body;
    }


    /**
     * 获取授权码
     *
     * @return accessToken
     * @throws BizException -
     */
    public String getToken() throws Exception {
        HttpPost httpPost = new HttpPost(wandaConfig.getQueryTokenUrl());
        List<NameValuePair> pairs = new ArrayList<>(4);
        Map<String, String> authParams = wandaConfig.getAuthParams();
        //租户号
        pairs.add(new BasicNameValuePair(COMPANY_CODE, authParams.get(COMPANY_CODE)));
        //授权类型
        pairs.add(new BasicNameValuePair(GRANT_TYPE, authParams.get(GRANT_TYPE)));
        //账号
        pairs.add(new BasicNameValuePair(ACCOUNT, authParams.get(ACCOUNT)));
        //秘钥
        pairs.add(new BasicNameValuePair(PASSWORD, authParams.get(PASSWORD)));
        httpPost.setEntity(new UrlEncodedFormEntity(pairs, StandardCharsets.UTF_8));
        //设置超时
        httpPost.setConfig(config);
        String response = "";
        try (CloseableHttpResponse httpResponse = httpClient.execute(httpPost)) {
            if (Objects.nonNull(httpResponse)) {
                response = EntityUtils.toString(httpResponse.getEntity(), StandardCharsets.UTF_8);
            }
        }
        //校验response
        if (StringUtils.isBlank(response)) {
            throw new BizException(WandaErrorCode.E1000001.getErrorMsg());
        }
        JSONObject result = JSONObject.parseObject(response);
        int status = result.getIntValue("status");
        String message = result.getString("message");
        //校验接口状态
        if (!Objects.equals(status, WandaRequestStatus.OK.getValue())) {
            throw new BizException(WandaErrorCode.E1000001.getErrorMsg() + "," + message);
        }
        JSONObject data = result.getJSONObject("data");
        if (Objects.nonNull(data)) {
            return data.getString(ACCESS_TOKEN);
        }
        return "";
    }

    /**
     * 获取用户当前积分
     *
     * @param cardNo 会员卡号（partner_user_id）
     * @return 用户积分
     */
    private Long queryCredits(String cardNo) throws Exception {
        String accessToken = getToken();
        LOGGER.info("[万达酒店]接口授权accessToken = [{}]", accessToken);
        HttpPost httpPost = new HttpPost(wandaConfig.getQueryCreditsUrl());
        //设置请求头
        httpPost.setHeader("Authentication", accessToken);
        List<NameValuePair> pairs = Collections.singletonList(new BasicNameValuePair("cardNo", cardNo));
        httpPost.setEntity(new UrlEncodedFormEntity(pairs, StandardCharsets.UTF_8));
        //设置超时
        httpPost.setConfig(config);
        String response = "";
        try (CloseableHttpResponse httpResponse = httpClient.execute(httpPost)) {
            if (Objects.nonNull(httpResponse)) {
                response = EntityUtils.toString(httpResponse.getEntity(), StandardCharsets.UTF_8);
            }
        }
        LOGGER.info("[万达酒店] 用户积分查询 = {}", response);
        //校验response
        if (StringUtils.isBlank(response)) {
            throw new BizException(WandaErrorCode.E1000001.getErrorMsg());
        }
        JSONObject result = JSONObject.parseObject(response);
        String responseCode = result.getString(RESPONSE_CODE);
        String message = result.getString("message");
        //校验接口状态
        if (!SUCCESS.equals(responseCode)) {
            throw new BizException(WandaErrorCode.E1000001.getErrorMsg() + "," + message);
        }
        JSONObject responseDomain = result.getJSONObject("responseDomain");
        if (Objects.nonNull(responseDomain)) {
            return responseDomain.getLong("point");
        } else {
            throw new BizException(WandaErrorCode.E1000003.getErrorMsg() + "," + message);
        }
    }



    /**
     * 截取域名
     *
     * @param url -
     * @return 域名
     */
    public static String getHostName(String url) {
        return url.substring(0, url.indexOf('?'));
    }

    public static String getParamUrl(String url) {
        return url.substring(url.indexOf('?') + 1);
    }

    /**
     * 获取酒店编码
     * 新增需求：在对方扣积分接口中需传酒店编码，若商品为兑吧商品或者活动抽奖，则酒店编码传默认值；
     * 若商品为开发者自有商品，优惠券与实物读取配置的商品编码参数，即兑吧原扣积分接口中的itemCode参数为酒店编码；
     * 若商品为虚拟商品，则读取配置的标志符解析，即兑吧原扣积分接口中的params参数解析，解析方式如下，params样式如：xxx_xxx，后端需通过下划线（“—”）分割解析，下划线前面为兑换券编码，下划线后面为酒店编码
     *
     * @param itemCode 商家商品编码
     * @return 酒店编码
     */
    @Deprecated
    private String getStoreCode(String itemCode, String type, boolean isDuibaItem) {
        final String splitter = "_";
        //没有itemCode(活动)
        if (StringUtils.isBlank(itemCode)) {
            return wandaConfig.getDefaultStoreCode();
        } else {
            //兑吧自有商品
            if (isDuibaItem) {
                return wandaConfig.getDefaultStoreCode();
            }
            //虚拟商品
            else if (VIRTUAL.equals(type)) {
                if (itemCode.contains(splitter)) {
                    String[] splitStrs = itemCode.split(splitter);
                    return Optional.ofNullable(splitStrs[1]).orElse("-1");
                } else {
                    throw new IllegalStateException("wanda虚拟商品 商家商品编码错误");
                }
            }
            //其他商品（开发者自有商品，优惠券与实物读取配置的商品）
            else {
                return itemCode;
            }
        }
    }
}
