/**
 * Project Name:tuia-core-api<br>
 * File Name:RemoteAdvertBackendService.java<br>
 * Package Name:cn.com.duiba.tuia.core.api.remoteservice<br>
 * Date:2016年8月19日下午4:22:04<br>
 * Copyright (c) 2016, duiba.com.cn All Rights Reserved.<br>
 */

package cn.com.duiba.tuia.core.api.remoteservice;

import cn.com.duiba.boot.exception.BizException;
import cn.com.duiba.boot.netflix.feign.AdvancedFeignClient;
import cn.com.duiba.tuia.core.api.dto.AdvertAbnormalConsumeDto;
import cn.com.duiba.tuia.core.api.dto.AdvertAbnormalCostDto;
import cn.com.duiba.tuia.core.api.dto.AdvertCheckRecordDto;
import cn.com.duiba.tuia.core.api.dto.AdvertDetailDto;
import cn.com.duiba.tuia.core.api.dto.AdvertDto;
import cn.com.duiba.tuia.core.api.dto.PageDto;
import cn.com.duiba.tuia.core.api.dto.advert.AdvertCouponInfoDto;
import cn.com.duiba.tuia.core.api.dto.advert.AdvertInfoDto;
import cn.com.duiba.tuia.core.api.dto.advert.AdvertValidStatusDto;
import cn.com.duiba.tuia.core.api.dto.advert.SupportNewTradeLaunchDto;
import cn.com.duiba.tuia.core.api.dto.advert.ValidAdvertInfoDto;
import cn.com.duiba.tuia.core.api.dto.advert.req.ReqPageQueryAdvertInfoDto;
import cn.com.duiba.tuia.core.api.dto.app.AdBaseInfoDto;
import cn.com.duiba.tuia.core.api.dto.req.ReqGetAdvertIds;
import cn.com.duiba.tuia.core.api.dto.req.ReqGetCreateAdvertDto;
import cn.com.duiba.tuia.core.api.dto.req.ReqPageQueryAdverts;
import cn.com.duiba.tuia.core.api.dto.req.ReqUpdateAdvertPlanDto;
import cn.com.duiba.tuia.core.api.dto.req.ReqUpdateCouponDto;
import cn.com.duiba.tuia.core.api.dto.req.advert.ReqAdvertDto;
import cn.com.duiba.tuia.core.api.dto.rsp.RspInsertCouponFileDto;
import cn.com.duiba.tuia.core.api.message.RedisMsgAnnotation;
import cn.com.duiba.wolf.dubbo.DubboResult;
import cn.com.tuia.advert.message.RedisMessageChannel;

import java.math.BigDecimal;
import java.util.List;
import java.util.Map;

/**
 * ClassName: RemoteAdvertBackendService <br/>
 * Function: 广告. <br/>
 * date: 2016年8月19日 下午4:22:04 <br/>
 *
 * @author leiliang
 * @since JDK 1.6
 */
@AdvancedFeignClient
public interface RemoteAdvertBackendService {

    /**
     * 创建广告.
     *
     * @param createAdvertDto the create advert dto
     * @return the dubbo result< long>
     */
    DubboResult<Long> createAdvert(ReqGetCreateAdvertDto createAdvertDto);

    /**
     * 上传优惠码文件.
     *
     * @param fileUrl the file url
     * @param fileName the file name
     * @return the dubbo result< long>
     */
    DubboResult<Long> uploadCouponCodeFile(String fileUrl, String fileName);

    /**
     * 更新启用状态.
     *
     * @param advertId 广告ID
     * @param status 启用状态(0：暂停， 1启用)
     * @return the dubbo result< boolean>
     */
    @RedisMsgAnnotation(value = {RedisMessageChannel.UPDATE_VALID_ADVERTS_MSG, RedisMessageChannel.UPDATE_ADVERTS_MSG})
    DubboResult<Boolean> updateEnableStatus(long advertId, int status);

    /**
     * 删除广告.
     *
     * @param advertId the advert id
     * @return the dubbo result< boolean>
     */
    DubboResult<Boolean> deleteAdvert(long advertId);

    /**
     * 单纯的删除接口，不包含业务逻辑
     *
     * @param advertId 删除的广告id
     * @return
     */
    DubboResult<Integer> simpleDeleteAdvert(Long advertId);

    /**
     * 更新广告计划.
     *
     * @param updateAdvertPlanDto the update advert plan dto
     * @return the dubbo result< boolean>
     */
    @RedisMsgAnnotation(value = {RedisMessageChannel.UPDATE_VALID_ADVERTS_MSG, RedisMessageChannel.UPDATE_ADVERTS_MSG, RedisMessageChannel.UPDATE_ADVERT_PERIOD})
    DubboResult<Boolean> updateAdvertPlan(ReqUpdateAdvertPlanDto updateAdvertPlanDto);

    /**
     * 更新广告
     *
     * @param updateAdvert
     * @return
     */
    DubboResult<Integer> updateAdvert(AdvertDto updateAdvert);

    /**
     * 更新广告优惠码数据.
     *
     * @param updateCouponDto the update coupon dto
     * @return the dubbo result< boolean>
     */
    @RedisMsgAnnotation(value = {RedisMessageChannel.UPDATE_ADVERTS_MSG, RedisMessageChannel.UPDATE_VALID_ADVERTS_MSG})
    DubboResult<Boolean> updateCoupon(ReqUpdateCouponDto updateCouponDto);

    /**
     * 更新审核状态.
     *
     * @param advertId 广告ID
     * @param status 审核状态(1：审核通过， 2：审核拒绝)
     * @param reason the reason
     * @param tagIds the tag ids
     * @return the dubbo result< boolean>
     */
    @RedisMsgAnnotation(value = {RedisMessageChannel.UPDATE_ADVERTS_MSG, RedisMessageChannel.UPDATE_VALID_ADVERTS_MSG})
    DubboResult<Boolean> updateCheckStatus(long advertId, int status, String reason, String tagIds);

    /**
     * 更新有效状态.
     *
     * @param advertId the advert id
     * @return the dubbo result< boolean>
     */
    @RedisMsgAnnotation(value = {RedisMessageChannel.UPDATE_ADVERTS_MSG, RedisMessageChannel.UPDATE_VALID_ADVERTS_MSG})
    DubboResult<Boolean> updateValidStatus(long advertId);

    /**
     * 更新广告有效状态
     *
     * @param advertId 更新的广告id
     * @return
     */
    DubboResult<Integer> simpleUpdateValidStatus(Long advertId, Integer validStatus);

    /**
     * 更新广告审核状态
     * @param advertId
     * @param checkStatus
     * @return
     */
    @Deprecated
    DubboResult<Integer> simpleUpdateCheckStatus(Long advertId, Integer checkStatus);
    
    /**
     * 更新广告审核状态
     * @param advertId
     * @param checkStatus
     * @param duibaAudit：兑吧审核状态
     * @return
     */
    Integer simpleUpdateAuditStatus(Long advertId, Integer checkStatus, Integer duibaAudi, String auditor);

    /**
     * 更新广告审核状态
     *
     * @param advertId     广告id
     * @param checkStatus  审核状态
     * @param refuseReason 审核原因
     */
    Integer updateAdvertAuditStatus(Long advertId, Integer checkStatus, String refuseReason, String auditor);

    /**
     * @param accountId 代理商id
     * @param advertId  广告id
     * @param name      广告名称
     * @return
     */
    DubboResult<Integer> countByIdAndName(Long accountId, Long advertId, String name);

    /**
     * 导入优惠码文件.
     *
     * @param advertId 广告ID
     * @param uploadRecordId 上传记录ID
     * @return the dubbo result< insert coupon file dto>
     */
    DubboResult<RspInsertCouponFileDto> insertCouponFile(long advertId, long uploadRecordId);

    /**
     * 查询广告详情.
     *
     * @param advertId 广告ID
     * @return the advert detail by id
     */
    DubboResult<AdvertDetailDto> getAdvertDetailById(long advertId);

    /**
     * 获取导入进展情况.
     *
     * @param importRecordId 导入记录ID
     * @return the import coupon count
     */
    DubboResult<RspInsertCouponFileDto> getImportCouponCount(long importRecordId);

    /**
     * 通过广告Id列表批量查询广告名称.
     *
     * @param advertIds 广告ID列表
     * @return 广告名称列表
     */
    DubboResult<List<AdvertDto>> getAdvertsByIds(List<Long> advertIds);

    List<AdvertDto> selectByIds(List<Long> ids);

    /**
     * 查询广告基本信息.
     *
     * @param advertId 广告
     * @return the advert base info by ids
     */
    DubboResult<AdvertDto> getAdvertById(long advertId);

    /**
     * 通过有效状态查询广告id列表
     *
     * @param validStatus 有效的状态
     * @return
     */
    DubboResult<List<Long>> selectIdsByValidStatus(Integer validStatus);

    /**
     * 通过广告名称和广告主id查询广告id
     *
     * @param advertName 广告名称
     * @param accountIds 广告主id
     * @return
     */
    DubboResult<List<Long>> selectIdsByNameAndAccountIds(String advertName, List<Long> accountIds);

    List<SupportNewTradeLaunchDto> selectNewTradeSupportLaunch();

    int updateNewTradeSupportLaunch(SupportNewTradeLaunchDto supportNewTradeLaunch);

    /**
     * 查询是否审核通过过.
     *
     * @param advertId 广告id
     * @return the dubbo result< boolean>
     */
    DubboResult<Boolean> isCheckPass(Long advertId);

    /**
     * 分页模糊查询列表.
     *
     * @param reqPageQueryAdverts the req page query adverts
     * @return the dubbo result< advert dto>
     */
    DubboResult<List<AdvertDto>> pageQuery(ReqPageQueryAdverts reqPageQueryAdverts);

    /**
     * 分页查询广告信息，不，这个方式是不分页的、不分页的、不分页
     * 若要分页请用pageQueryAdvertList方法
     * @param advertInfo
     * @return
     */
    DubboResult<List<AdvertInfoDto>> pageQueryAdvertsInfo(ReqPageQueryAdvertInfoDto advertInfo);

    /**
     * 分页查询广告列表，这个方法是分页的、分页的
     */
    List<AdvertInfoDto> pageQueryAdvertList(ReqPageQueryAdvertInfoDto advertInfo);

    /**
     * 给分页查询的总数统计接口
     *
     * @param advertInfo
     * @return
     */
    DubboResult<Integer> countQueryAdvertsInfo(ReqPageQueryAdvertInfoDto advertInfo);

    /**
     * 分页模糊查询数量.
     *
     * @param reqPageQueryAdverts the req page query adverts
     * @return the dubbo result< integer>
     */
    DubboResult<Integer> pageQueryAmount(ReqPageQueryAdverts reqPageQueryAdverts);

    /**
     * 批量查询广告的最近的审核记录.
     *
     * @param advertIds 广告ID列表
     * @return 最近的审核记录
     */
    DubboResult<List<AdvertCheckRecordDto>> getLatestCheckRecords(List<Long> advertIds);

    /**
     * 查询有效广告Id列表.
     *
     * @return the list< long>
     */
    DubboResult<List<Long>> selectValiAdvertIds();

    /**
     * 查询需要数据修复的广告ID集合
     *
     * @return 广告ID
     */
    List<AdvertDto> queryNeedFixAdvertList();

    /**
     * 更新广告单价.
     *
     * @param advertId 广告id
     * @param fee 广告单价
     * @return the dubbo result< boolean>
     */
    @RedisMsgAnnotation(value = {RedisMessageChannel.UPDATE_ADVERTS_MSG, RedisMessageChannel.UPDATE_VALID_ADVERTS_MSG})
    DubboResult<Boolean> updateFee(long advertId, int fee);

    /**
     * 更新广告每日预算.
     *
     * @param advertId 广告id
     * @param budgetPerDay 广告每日预算
     * @return the dubbo result< boolean>
     */
    @RedisMsgAnnotation(value = {RedisMessageChannel.UPDATE_ADVERTS_MSG, RedisMessageChannel.UPDATE_VALID_ADVERTS_MSG})
    DubboResult<Boolean> updateBudgetPerDay(long advertId, Integer budgetPerDay);

    /**
     * 查询所有满足条件的广告Id列表.<br>
     * [查询条件]
     * <ol>
     * <li></li>
     * </ol>
     *
     * @param req the req
     * @return the advertiser advert ids
     */
    DubboResult<List<AdvertDto>> getAdvertIdsOfAdvertiser(ReqGetAdvertIds req);

    List<AdvertDto> getNeedFixAdvertiserAdvertList(ReqGetAdvertIds req);

    /**
     * 模糊查询广告主下的所有广告列表.<br>
     * <ol>
     * <li>根据advertName模糊查询</li>
     * <li>返回字段：广告ID、广告名称</li>
     * </ol>
     *
     * @param advertiserId the advertiser id
     * @param advertName 广告名称
     * @return the list by agent id
     */
    DubboResult<List<AdvertDto>> getListByAdvertiserId(long advertiserId, String advertName);

    /**
     * 模糊查询代理商下的所有广告列表.<br>
     * <ol>
     * <li>根据advertName模糊查询</li>
     * <li>返回字段：广告ID、广告名称</li>
     * </ol>
     *
     * @param agentId 代理商ID
     * @param advertName 广告名称
     * @return the list by agent id
     */
    DubboResult<List<AdvertDto>> getListByAgentId(long agentId, String advertName);

    /**
     * 模糊查询操作员下所有广告列表.<br>
     * <ol>
     * <li>根据advertName模糊查询</li>
     * <li>返回字段：广告ID、广告名称</li>
     * </ol>
     *
     * @param operatorId the operator id
     * @param advertName 广告名称
     * @return the list by operate id
     */
    DubboResult<List<AdvertDto>> getListByOperatorId(long operatorId, String advertName);

    /**
     * 查询广告列表分页信息.
     *
     * @param reqPageQueryAdverts the req page query adverts
     * @return the dubbo result< page dto< list< advert dto>>>
     */
    DubboResult<PageDto<AdvertDto>> selectAdvertPageList(ReqPageQueryAdverts reqPageQueryAdverts);

    /**
     * 代理商登录查询广告列表.
     *
     * @param agentId the agent id
     * @param advertName the advert name
     * @param email the email
     * @param companyname the companyname
     * @return the list by agent
     */
    DubboResult<List<AdvertDto>> getListByAgentLogin(long agentId, String advertName, String email, String companyname);

    List<AdvertDto> getFixAdvertListByAgent(long agentId, String advertName, String email, String companyName);

    /**
     * 初始化ES数据
     *
     * @return
     */
    DubboResult<Integer> initEsData(Boolean reIndex);

    /**
     * 更新广告数据（重新发布广告）
     *
     * @param advertId  需要更新的广告id
     * @param accountId 代理商id
     * @param name      广告名称
     * @return
     */
    DubboResult<Integer> updateDuibaAdvertById(Long advertId, Long accountId, String name);

    /**
     * 通过accountId更新有效状态
     *
     * @param accountId   需要更新的代理商id
     * @param validStatus 更新的状态
     * @return
     */
    DubboResult<Integer> updateAdvertValidStatusByAccountId(Long accountId, Integer validStatus);

    /**
     * 查询代理商下的所有广告
     *
     * @param accountId 查询的代理商id
     * @return
     */
    DubboResult<List<AdvertDto>> listAllAdvertsByAccountId(Long accountId);

    /**
     * 查询广告主下面当前在投的有效广告
     *
     * @return
     */
    List<AdvertDto> selectAccountValidAdvertList(List<Long> accountIds);

    /**
     * 根据广告名称查询有效的广告信息
     *
     * @param advertName 匹配的广告名称
     * @param rowStart   分页参数
     * @param pageSize   分页参数
     * @return
     */
    DubboResult<List<ValidAdvertInfoDto>> pageListValidAdvertsByName(String advertName, Integer rowStart, Integer pageSize);

    /**
     * 根据广告名称计数有效的广告信息数量
     *
     * @param advertName 匹配的广告名称
     * @return
     */
    DubboResult<Integer> pageCountValidAdvertsByName(String advertName);

    /**
     * 分页查询的统计计数接口
     *
     * @param accountId 广告主id
     * @param advertId  广告id
     * @return
     */
    DubboResult<Integer> pageCountByAccountIdAndAdvertId(Long accountId, Long advertId);

    /**
     * 更具广告主id分页查询数据
     *
     * @param accountId 广告主id
     * @param advertId
     * @param rowStart
     * @param pageSize
     * @return
     */
    DubboResult<List<AdvertDto>> pageQueryByAccountIdAndAdvertId(Long accountId, Long advertId, Integer rowStart, Integer pageSize);
    /**
     * 根据id更新排序级别
     *
     * @param advertId        需要更新的广告id
     * @param level           排序级别
     * @param levelUpdateTime 级别更新时间
     * @return
     */
    DubboResult<Integer> updateAdvertLevelById(Long advertId, Long level, Long levelUpdateTime);

    /**
     * 保存异常提醒开关
     *
     * @param invalidRemindStatus    广告失效提醒，1-开启，0-关闭
     * @param consumeAbnRemindStatus 消耗异常提醒，1-开启，0-关闭
     */
    Boolean saveAbnormalRemind(Integer invalidRemindStatus, Integer consumeAbnRemindStatus, Long advertId);
    /**
     * 查询广告列表
     *
     * @param accountId   代理商id
     * @param validStatus 有效状态
     * @return
     */
    DubboResult<List<AdvertDto>> listAdvertsByAccountIdAndValidStatus(Long accountId, Integer validStatus);

    /**
     * 通过source查询数据
     *
     * @param sources
     * @return
     */
    DubboResult<List<AdvertDto>> selectDuibaIdBySource(List<Integer> sources);

    /**
     * 更具广告id批量更新有效的状态
     *
     * @param updateValidStatusList 需要更新的数据
     * @return 更新数量
     */
    DubboResult<Integer> batchUpdateAdvertValidStatus(List<AdvertValidStatusDto> updateValidStatusList);

    /**
     * 通过duibaId查询数据
     *
     * @param source
     * @param duibaId 对吧id
     * @return
     */
    DubboResult<AdvertDto> selectAdvertByDuibaId(Integer source, Long duibaId);

    /**
     * 添加广告
     *
     * @param advert
     * @return
     */
    DubboResult<Integer> saveAdvert(AdvertDto advert);

    /**
     * 根据广告名称模糊查询广告id
     *
     * @param advertName 广告名称
     * @return 广告id列表
     */
    DubboResult<List<Long>> getAdvertIdsByAdvertName(String advertName);

    DubboResult<List<Long>> getAdvertIdsByAgentIds(List<Long> agentIds);

    DubboResult<List<Long>> getAdvertIdsByAccountIds(List<Long> accountIds);

    /**
     * 更新广告权重
     * 
     * @param advertId
     * @param advertWeight
     * @return
     */
    DubboResult<Boolean> updateAdvertWeight(long advertId, BigDecimal advertWeight);
    
    /**
     * 注意：最好统一用selectAdvertList方法
     * selectAdvertsByAccountId:(根据广告主条件查询广告信息). <br/>
     * 
     * @author chencheng
     * @param accountId 广告主id必须
     * @param isDeleted 是否被删除 ,1-已删除;0-未删除
     * @param checkStatus 审核状态,0-审核中;1-审核通过;2-审核拒绝
     * @param validStatus 有效状态 @{AdvertValidStatusEnum}
     * @param source 广告来源,0-广告平台,1-兑吧商品,2-券库
     * @param enableStatus 启用状态,1-启用;0-暂停
     * @return
     * @throws TuiaCoreException 
     * @since JDK 1.8
     */
    @Deprecated
    DubboResult<List<AdvertDto>> selectAdvertsByAccountId(AdvertDto advertDto);
    
    /**
     * 
     * selectAdvertList:(根据条件查询广告信息). <br/>
     *
     * @author chencheng
     * @param req
     * @return
     * @throws TuiaCoreException 
     * @since JDK 1.8
     */
    List<AdvertDto> selectAdvertList(ReqAdvertDto req) throws Exception;

    /**
     * 调用商品中心批量查询广告券列表接口
     *
     * @param ids duibaIds
     * @return 广告券列表
     */
    List<AdvertCouponInfoDto> findCouponByAcgIds(List<Long> ids);

    /**
     * 查询广告的基础信息
     *
     * @param advertIds 查询的广告ID列表
     * @return 改接口广告的配置列表不会返回
     */
    List<AdBaseInfoDto> getAdvertBaseInfoByIds(List<Long> advertIds);

    /**
     * 查询广告信息和广告的配置包信息
     *
     * @param advertIds 查询的广告id
     */
    List<AdBaseInfoDto> queryBaseInfoByAdvertIds(List<Long> advertIds);

    Boolean enableCostNotice(Long advertId, Long consume, Long cost);
    Boolean disableCostNotice(Long advertId);

    AdvertAbnormalCostDto getCostInfo(Long advertId);

    AdvertAbnormalConsumeDto getConsumeInfo(Long advertId);

    Boolean insertConsumeInfo(Long advertId, Long yesterdayConsume, Double fluctuateRate);

    Boolean updateConsumeInfo(Long advertId, Long yesterdayConsume, Double fluctuateRate);

    /**
     * 
     * updateAdvertById:(这里用一句话描述这个方法的作用). <br/>
     *
     * @author chencheng
     * @param advertDto
     * @return
     * @since JDK 1.8
     */
    Integer updateAdvertById (AdvertDto advertDto);

    /**
     * 根据advertId或者name查询advertId
     * @param advertIdName
     * @return
     */
    Long findAdvertIdByAdvertIdOrAdvertName(String advertIdName);

    /**
     * 根据advertiserId或者name或者account获取advertIds
     * @param advertiserIdNameAccount
     * @return
     */
    List<Long> findAdvertIdsByAdvertiserIdNameAccount(String advertiserIdNameAccount);

    /**
     * 根据代理商ID，名称，帐号获取旗下的广告ids
     * @param agentIdNameAccount
     * @return
     */
    List<Long> findAdvertIdsByAgentIdNameAccount(String agentIdNameAccount);

    /**
     * 根据代理商邮箱查询其下面所有广告主id
     *
     */
    List<Long> selectAdvertiserIdsByAgentEmail(String agentEmail);

    /**
     * 查询出所有广告id
     * @return
     */
    List<Long> selectAllAdvertIds();

    /**
     * 根据某个广告主，以及广告id或者广告名称查询
     */
    List<AdvertDto> selectAdvertListByAccount(Long AdvertiserId,Long advertId ,String AdvertName);

    /**
     * 根据广告id 和广告名称 查询一批广告
     */
    List<AdvertDto> selectAdvertListByAdvertIds(List<Long> advertIds,String advertName);

    /**
     * 根据落地页链接查询关联的广告id集合
     */
    List<Long> selectAdvetIdsByLink(String link);

    /**
     * 修改广告关联的 落地页链接
     */
    void updateAdvertPromoteUrl(Map<Long,String> promoteUrlMap) throws BizException;

    /**
     * 广告 - 考核指标
     * @return
     */
    Map<Long, Integer> selectAssessByAdvertIds(List<Long> advertIds);

    /**
     * 根据广告ids 查询 广告名称
     * @param advertIds
     * @return
     */
    List<AdBaseInfoDto> selectAdvertNameByAdvertIds(List<Long> advertIds);
}
