package cn.com.duiba.tuia.adx.center.api.common;

import cn.com.duiba.tuia.adx.center.api.constant.ColumnName;
import lombok.Data;
import org.springframework.lang.Nullable;
import org.springframework.util.Assert;

import java.io.Serializable;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * @author ikongyi
 */
@Data
public class OrderBy implements Serializable {
    private static final long serialVersionUID = 3200046027483541175L;

    public static final Direction DEFAULT_DIRECTION = Direction.ASC;

    /**
     * 列名
     * @see ColumnName
     */
    private String columnName;

    /**
     * sort directions
     */
    private Direction direction;

    /**
     * Creates a new {@link OrderBy} instance. Takes a single property. Direction defaults to {@link #DEFAULT_DIRECTION}.
     *
     * @param columnName must not be {@literal null}
     */
    public static OrderBy by(String columnName) {
        return newOrderBy(DEFAULT_DIRECTION, columnName);
    }

    /**
     *
     * @param columnNames must not be {@literal null}.
     * @return {@link List<OrderBy>}
     */
    public static List<OrderBy> by(String... columnNames) {
        Assert.notNull(columnNames, "columnNames must not be null!");
        Assert.isTrue(columnNames.length > 0, "At least one columnName must be given!");

        return Arrays.stream(columnNames).map(OrderBy::by).collect(Collectors.toList());
    }

    /**
     *
     * @param direction must not be {@literal null}.
     * @param columnNames must not be {@literal null}.
     * @return
     */
    public static List<OrderBy> by(Direction direction, String... columnNames) {
        Assert.notNull(direction, "Direction must not be null!");
        Assert.notNull(columnNames, "columnNames must not be null!");
        Assert.isTrue(columnNames.length > 0, "At least one columnName must be given!");

        return Arrays.stream(columnNames).map(it -> newOrderBy(direction, it)).collect(Collectors.toList());
    }

    /**
     * Creates a new {@link OrderBy} instance. Takes a single property. Direction is {@link Direction#ASC}
     *
     * @param columnName must not be {@literal null}
     */
    public static OrderBy asc(String columnName) {
        return newOrderBy(Direction.ASC, columnName);
    }

    /**
     * Creates a new {@link OrderBy} instance. Takes a single property. Direction is {@link Direction#DESC}
     *
     * @param columnName must not be {@literal null}
     */
    public static OrderBy desc(String columnName) {
        return newOrderBy(Direction.DESC, columnName);
    }

    /**
     * Creates a new {@link OrderBy} instance. if order is {@literal null} then order defaults to
     * {@link #DEFAULT_DIRECTION}
     *
     * @param direction can be {@literal null}, will default to {@link #DEFAULT_DIRECTION}
     * @param columnName must not be {@literal null}
     */
    static OrderBy newOrderBy(@Nullable Direction direction, String columnName) {
        if (Objects.isNull(columnName)) {
            throw new IllegalArgumentException("Property must not null!");
        }
        OrderBy orderBy = new OrderBy();
        orderBy.setColumnName(columnName);
        orderBy.setDirection(direction == null ? DEFAULT_DIRECTION : direction);
        return orderBy;
    }

    /*
     * (non-Javadoc)
     * @see java.lang.Object#hashCode()
     */
    @Override
    public int hashCode() {
        int result = 17;

        result = 31 * result + direction.hashCode();
        result = 31 * result + columnName.hashCode();

        return result;
    }

    /*
     * (non-Javadoc)
     * @see java.lang.Object#equals(java.lang.Object)
     */
    @Override
    public boolean equals(@Nullable Object obj) {
        if (this == obj) {
            return true;
        }

        if (!(obj instanceof OrderBy)) {
            return false;
        }

        OrderBy that = (OrderBy) obj;

        return this.direction.equals(that.direction) && this.columnName.equals(that.columnName);
    }

    /*
     * (non-Javadoc)
     * @see java.lang.Object#toString()
     */
    @Override
    public String toString() {
        return String.format("%s: %s", columnName, direction);
    }
    
    /**
     * Enumeration for sort directions.
     */
    public enum Direction {

        ASC, DESC;

        /**
         * Returns whether the direction is ascending.
         */
        public boolean isAscending() {
            return this.equals(ASC);
        }

        /**
         * Returns whether the direction is descending.
         */
        public boolean isDescending() {
            return this.equals(DESC);
        }
    }
}
