package cn.com.duiba.tuia.youtui.center.api.util;

import cn.com.duiba.tuia.youtui.center.api.dto.youtui.BaseRatioDto;
import cn.com.duiba.tuia.youtui.center.api.dto.youtui.IRatioAble;
import com.alibaba.fastjson.JSON;
import com.google.common.collect.Lists;

import java.util.Collections;
import java.util.List;
import java.util.Random;
import java.util.stream.Collectors;

/**
 * 随机、分流工具类
 */
public class RandomFlowUtil {

    /**
     * 从list中随机获取一个元素
     *
     * @param list
     * @param <T>
     * @return
     */
    public static <T> T random4List(List<T> list) {
        if (list == null) {
            return null;
        }
        int size = list.size();
        if (size == 0) {
            return null;
        }
        int index = new Random().nextInt(size);
        return list.get(index);
    }

    /**
     * 获取100以内的随机整数
     *
     * @return
     */
    public static int randomInHundred() {
        return new Random().nextInt(100);
    }

    /**
     * 判断百分比
     * 如： randomPer(20)，表示20%的概率
     *
     * @param per 期望百分比
     * @return 是否在这个百分比范围内
     */
    public static boolean randomPer(int per) {
        int random = randomInHundred();
        return random <= per;
    }

    /**
     * 判断百分比
     *
     * @param min 下限
     * @param max 上限
     * @return
     */
    public static boolean randomPer(int min, int max) {
        int random = new Random().nextInt(100);
        return randomPer(random, min, max);
    }

    /**
     * 判断百分比
     *
     * @param min 下限
     * @param max 上限
     * @return
     */
    public static boolean randomPer(int per, int min, int max) {
        return per <= max && per > min;
    }

    /**
     * 计算ID是否在指定区间
     *
     * @param id
     * @param min
     * @param max
     * @return
     */
    public static boolean randomPerById(Long id, int min, int max) {
        Integer per = computePerWithId(id);
        return per <= max && per > min;
    }

    /**
     * 根据ID计算百分比
     *
     * @param id
     * @return
     */
    public static Integer computePerWithId(Long id) {
        if (id == null) {
            return 0;
        }
        return Long.valueOf(id % 99).intValue();
    }

    /**
     * 根据ID计算对应的概率对象
     *
     * @param id
     * @param baseRatioDtos
     * @return index,-1表示异常
     */
    public static <T extends BaseRatioDto> Integer computeIndexById(Long id, List<T> baseRatioDtos) {
        List<RatioTarget> ratioTargets = init(baseRatioDtos);
        Integer per = computePerWithId(id);
        for (RatioTarget rt : ratioTargets) {
            if (randomPer(per, rt.head.getRatio(), rt.getRatio())) {
                return rt.getIndex();
            }
        }
        //极小概率事件，找不对对应概率，就随机出一个
        return new Random().nextInt(baseRatioDtos.size());
    }

    /**
     * 随机数计算对应的概率对象
     *
     * @param baseRatioDtos
     * @return
     */
    public static <T extends BaseRatioDto> Integer computeIndex(List<T> baseRatioDtos) {
        List<RatioTarget> ratioTargets = init(baseRatioDtos);
        int random = randomInHundred();
        for (RatioTarget rt : ratioTargets) {
            if (randomPer(random, rt.head.getRatio(), rt.getRatio())) {
                return rt.getIndex();
            }
        }
        //极小概率事件，找不对对应概率，就随机出一个
        return new Random().nextInt(baseRatioDtos.size());
    }

    /**
     * 随机数计算对应的概率对象
     *
     * @param iBaseRatios
     * @return
     */
    public static <T extends IRatioAble> T computeRatioAble(List<T> iBaseRatios) {
        List<BaseRatioDto> baseRatioDtos = Lists.newArrayList();
        for(int i=0;i<iBaseRatios.size();i++){
            T t = iBaseRatios.get(i);
            BaseRatioDto dto = new BaseRatioDto(t.getRatio(),i);
            baseRatioDtos.add(dto);
        }
        Integer integer = computeIndex(baseRatioDtos);
        return iBaseRatios.get(integer);
    }

    /**
     * 生成百分比list
     *
     * @param list
     * @return
     */
    public static <T extends BaseRatioDto> List<RatioTarget> init(List<T> list) {
        if (list == null) {
            return null;
        }
        List<T> listCopy = Lists.newArrayList(list);
        if(listCopy.size()>1){
            listCopy.sort(BaseRatioDto::compare);
        }
        List<RatioTarget> ratioTargets = Lists.newArrayList();
        Integer count = 0;
        for (BaseRatioDto dto : listCopy) {
            count += dto.getRatio();
        }
        for (int i = 0; i < listCopy.size(); i++) {
            BaseRatioDto ratioDto = listCopy.get(i);
            if (i == 0) {
                RatioTarget ratioTarget = new RatioTarget(new RatioTarget(0), i, ratioDto.getRatio() * 100 / count);
                ratioTargets.add(ratioTarget);
            } else {
                RatioTarget head = ratioTargets.get(i - 1);
                RatioTarget ratioTarget = new RatioTarget(head, i, head.getRatio() * 100 / count + ratioDto.getRatio() * 100 / count);
                ratioTargets.add(ratioTarget);
            }
        }
        return ratioTargets;
    }

    /**
     * 概率对象
     */
    public static class RatioTarget {

        private RatioTarget head;

        private Integer index;

        private Integer ratio;

        public RatioTarget() {
        }

        public RatioTarget(Integer ratio) {
            this.ratio = ratio;
        }

        public RatioTarget(RatioTarget head, Integer index, Integer ratio) {
            this.head = head;
            this.index = index;
            this.ratio = ratio;
        }

        public Integer getIndex() {
            return index;
        }

        public void setIndex(Integer index) {
            this.index = index;
        }

        public Integer getRatio() {
            return ratio;
        }

        public void setRatio(Integer ratio) {
            this.ratio = ratio;
        }
    }

    public static void main(String[] args) {
        List<BaseRatioDto> list = Lists.newArrayList();
        BaseRatioDto baseRatioDto1 = new BaseRatioDto();
        baseRatioDto1.setIndex(1);
        baseRatioDto1.setRatio(10);
        list.add(baseRatioDto1);
        BaseRatioDto baseRatioDto2 = new BaseRatioDto();
        baseRatioDto2.setIndex(2);
        baseRatioDto2.setRatio(30);
        list.add(baseRatioDto2);
        BaseRatioDto baseRatioDto3 = new BaseRatioDto();
        baseRatioDto3.setIndex(3);
        baseRatioDto3.setRatio(60);
        list.add(baseRatioDto3);
        System.err.println(JSON.toJSONString(init(list)));
    }
}
